<?php
/**
* class JsSlideshow
*
* Used for simple css/html slideshow creation
*
* @package   Main
* @version   0.02
* @since     2015-03-24
*
* @example ../expressiontest.php
* @example ../lifepathtest.php
* @example <br />
*   $slideshow = new JsSlideshow("bg.jpg", 640, 480);<br />
*   $slideshow->addAudio( $audioFile);<br />
*   $slideshow->addSlide( $slideTime );<br />
*   $slideshow->setFontSize( $fontSize );<br />
*   $slideshow->setTitle( $title );<br />
*   $slideshow->addTiles( $fullBirthName ); <br />
*   $slideshowpath = $slideshow->compile();
*/

class JsSlideshow
{

    /**
    * Array with slides
    * @access public
    * @var array
    */
    public $slides = array();

    /**
    * Array with replaces
    * @access private
    * @var array
    */
    private $replaces = array('from'=>array(),'to'=>array());


    /**
    * Tab width for Bullets in persrnt
    * @access public
    * @var integer
    */
    public $tabWidth = 40;


    /**
    * Path to directory with tiles images
    * @access private
    * @var string
    */
    private $tilesImagesDirectory = "images";

    /**
    * Path where slides to be saved
    * @access private
    * @var string
    */
    private $slidesOutputPath = 'slides';

    /**
    * Path to ffmpeg executable file
    * @access private
    * @var string
    */
    private $ffmpegBinaryPath = "usr/bin/ffmpeg";

    /**
    * Array with audio files
    * @access protected
    * @var array
    */
    protected $slideshowSounds = array();
    
    /**
    * Array with js scripts
    * @access protected
    * @var array
    */
    protected $slideshowScripts = array();

    /**
    * Current slide index
    * @access protected
    * @var integer
    */
    protected $currentSlideIndex = null;

    /**
    * Global slideshow css styles object
    * @access protected
    * @var stdClass
    */
    protected $slideshowStyles;

    /**
    * Total duration of all audio files in slideshow
    * @access protected
    * @var float
    */
    protected $slideshowPlaytime = 0;

    /**
    * Total duration of all slides in slideshow
    * @access protected
    * @var float
    */
    protected $slideshowSlidesTime = 0;

    /**
    * Unique ID of slideshow
    * @access protected
    * @var string
    */
    protected $slideshowID;

    /**
    * Internal flag that indicates whether to clean "canvas"
    * @access protected
    * @var boolean
    */
    protected $clearSlide = false;


    /**
    * Google Cloud Storage variables
    * @access public
    */
    
	public $gcs = ['client', 'credential', 'storage', 'projectId' => 'numerologist-888', 'fileName', 'ownerAccess', 'readerAccess', 'obj', 'url' => 'https://storage.googleapis.com/numerologist-888.appspot.com/', 'initialized' => false];
	

    /**
    * Create an instance
    *
    * @param string  Path to the background image file
    * @param integer (optional) Width of slideshow
    * @param integer (optional) height of slideshow
    * @param string (optional) Unique id of slideshow
    *
    * @access public
    */
    public function __construct($backgroundFile, $width = null, $height = null, $id = null)
    {
        require_once(dirname(__FILE__)."/getid3/getid3.php");

        $this->slideshowStyles = new stdClass();
        $this->slideshowStyles->background = $backgroundFile;
        $this->slideshowStyles->width = $width;
        $this->slideshowStyles->height = $height;
        $this->slideshowID = isset($id) ? $id : self::uniqID();
        
    }

    /**
    * Initializes Google Cloud Storage (using Google Cloud PHP API) for file insertion.  Only needs to be done once for object instance.
    *
    * @param void
    *
    * @access public
    * @return void
    */

	public function gcsInitialize() {

		$this->gcs['client'] = new \Google_Client();
		$this->gcs['client']->setApplicationName("GCS Upload");
		$this->gcs['credential'] = new \Google_Auth_AssertionCredentials(
						"numerologist-888@appspot.gserviceaccount.com",
						['https://www.googleapis.com/auth/devstorage.read_write'],
						file_get_contents(dirname(__FILE__)."/Google/Keys/Numerologist-783fe9fe3f61.p12")
					);
		$this->gcs['client']->setAssertionCredentials($this->gcs['credential']);
		if($this->gcs['client']->getAuth()->isAccessTokenExpired()) {
			$this->gcs['client']->getAuth()->refreshTokenWithAssertion($this->gcs['credential']);
		}		
		
		$this->gcs['ownerAccess'] = new \Google_Service_Storage_ObjectAccessControl();
		$this->gcs['ownerAccess']->setEntity('project-owners-' . $this->gcs['projectId']);
		$this->gcs['ownerAccess']->setRole('OWNER');
		
		$this->gcs['readerAccess'] = new \Google_Service_Storage_ObjectAccessControl();
		$this->gcs['readerAccess']->setEntity('allUsers');
		$this->gcs['readerAccess']->setRole('READER');			

		$this->gcs['initialized'] = true;									        		
	}

	public function gcsInsert($fileName, $mimeType = 'audio/mpeg') {
		
		if(!$this->gcs['initialized']) {
			$this->gcsInitialize();
		}
		
		$this->gcs['storage'] = new \Google_Service_Storage($this->gcs['client']);	
		
		$this->gcs['obj'] = new \Google_Service_Storage_StorageObject();
		$this->gcs['obj']->setName($fileName);
		$this->gcs['obj']->setAcl([$this->gcs['ownerAccess'], $this->gcs['readerAccess']]);
		
		
		$this->gcs['storage']->objects->insert(
		    $this->gcs['projectId'].'.appspot.com',
		    $this->gcs['obj'],
		    [
		        'data'       => file_get_contents($fileName),
		        'uploadType' => 'media',
		        'mimeType' => $mimeType
		    ]
		);
	}

    /**
    * Sets the tiles images directory
    *
    * @param string  Path to the tiles images directory
    *
    * @access public
    * @return void
    */
    public function setTilesImagesDir($path)
    {
        $this->tilesImagesDirectory = $path;
    }


    /**
    * Sets the directory where the slides will be saved
    *
    * @param string  Path to the slides directory
    *
    * @access public
    * @return void
    */
    public function setSlidesPath($path)
    {
        $this->slidesOutputPath = $path;
    }


    /**
    * Sets the path to ffmpeg executable file
    *
    * @param string  Path to the slides directory
    *
    * @access public
    * @return void
    */
    public function setFfmpegBin($path)
    {
        $this->ffmpegBinaryPath = $path;
    }


    /**
    * Sets the Tab width
    *
    * @param integer Tab witdth in persent
    *
    * @access public
    * @return void
    */
    public function setTab($width)
    {
        $this->tabWidth = $width;
    }

    /**
    * Get slide object from index or current slide
    *
    * @param integer Slide index
    * @access public
    * @return stdClass|boolean Slide object or FALSE
    */
    public function get($index = null)
    {
        if (!$index) {
            $index = $this->currentSlideIndex;
        }
        if (isset($this->slides[$index])) {
            $this->currentSlideIndex = $index;
            return $this->slides[$index];
        } else {
            return false;
        }
    }

    /**
    * Get slideshow id
    *
    * @access public
    * @return string
    */
    public function getID()
    {
        return $this->slideshowID;
    }

    /**
    * Adds a slide object to the slideshow
    * if no time is specif ied, the slide will be displayed until
    * the end of the audio file
    *
    * @param float Time in seconds
    * @access public
    * @return JsSlideshow
    */
    public function addSlide($time = null)
    {
        if ($this->slideshowPlaytime == 0) {
            throw new Exception("Add the sound first");
        }

        if (!isset($time)) {
            $time = $this->slideshowPlaytime - $this->slideshowSlidesTime;
        }

        if ($this->slideshowSlidesTime > $this->slideshowPlaytime) {
            throw new Exception("Slideshow duration more than the duration of the all sounds. 
                                Add more sounds or set a shorter duration of the slide");
        }

        $slide = new stdClass();
        $slide->clear = $this->clearSlide;
        $slide->items = array();
        $slide->time = $time;
        $slide->css = array();
        $this->slideshowSlidesTime += $time;

        $this->slides[] = $slide;
        $this->currentSlideIndex = count($this->slides)-1;
        $this->clearSlide = false;
        return $this;
    }

    /**
    * Returns the length of the audio file at the specif ied index
    * or length of the last audio file is added to the slideshow
    * if no index is specif ied
    *
    * @param integer|null (optional) Index of audio file in slideshow
    * @access public
    * @return float Lenght of audio file
    */
    public function getAudioLen($index = null)
    {
        if (!isset($index)) {
            $index = count($this->slideshowSounds) - 1;
        }
        return isset($this->slideshowSounds[$index]) ? $this->slideshowSounds[$index]['duration'] : 0;
    }

    /**
    * Adds a title to the slide
    *
    * @example $slideshow->setTitle( "Hello there" ); //Simple
    * @example $slideshow->setTitle( 'Hello <red>there</red>', ['margin-top'=>'50px'] );
    *          //Available all power of CSS markup
    *
    * @access public
    *
    * @param string Title text
    * @param array (optional) CSS styles
    *
    * @return JsSlideshow
    */
    public function setTitle($title, $style = null)
    {
        $slide = $this->get($this->currentSlideIndex);
        $item = array();
        $item['type'] = 'title';
        $item['html'] = '<p class="title">'.self::parseTags(str_replace($this->replaces['from'], $this->replaces['to'], $title)).'</p>';
        $item['style'] = isset($slide->style['fontSize']) ? array('fontSize'=>$slide->style['fontSize']) : array();
        if (isset($style) && is_array($style)) {
            $item['style'] = array_merge($item['style'], $style);
        }
        $slide->items[] = $item;
        return $this;
    }


    /**
    * Adds a any HTML code to the slide
    *
    * @example $slideshow->addHTML( '<div class="name"> ... </div>' ); //Simple
    * @example $slideshow->addHTML( '<div class="name"> ... </div>', ['margin-top'=>'50px'] );
    *          //Available all power of CSS markup
    *
    * @access public
    *
    * @param string HTML code
    * @param array (optional) CSS styles
    *
    * @return JsSlideshow
    */
    public function addHTML($html, $style = null)
    {
        $slide = $this->get($this->currentSlideIndex);
        $item = array();
        $item['type'] = 'html';
        $item['html'] = '<div class="html">'.self::parseTags(str_replace($this->replaces['from'], $this->replaces['to'], $html)).'</div>';
        $item['style'] = isset($slide->style['fontSize']) ? array('fontSize'=>$slide->style['fontSize']) : array();
        if (isset($style) && is_array($style)) {
            $item['style'] = array_merge($item['style'], $style);
        }
        $slide->items[] = $item;
        return $this;
    }

    /**
    * Adds a javascript callback, which will be called when the slide shown
    *
    * @example //Call JavascriptFunctionName() in frontend when slide is showing<br />
    *          $slideshow->addCallback( 'JavascriptFunctionName' );
    *
    * @example //Call alert('Hello Alex!') in frontend when slide is showing<br />
    *          $slideshow->addCallback( 'alert', ['Hello Alex!'] );
    *
    * @access public
    *
    * @param string Callback function name
    * @param array (optional) Arguments, which passed to the function
    *
    * @return JsSlideshow
    */
    public function addCallback($fn, $arguments = array())
    {
        $slide = $this->get($this->currentSlideIndex);
        $item = array();
        $item['type'] = 'callback';
        $item['fn'] = $fn;
        $item['arguments'] = $arguments;
        $slide->items[] = $item;
        return $this;
    }


    /**
    * Adds a Bullet to the slide
    *
    * @example $slideshow->addBullet( "Hello there" ); //Simple
    * @example $slideshow->addBullet( 'First slide:\tHello <red>there</red>');
    *          //Bullet with tab
    * @example $slideshow->addBullet( 'Hello <red>there</red>', ['margin-top'=>'50px'] );
    *          //Available all power of CSS markup
    *
    * @access public
    *
    * @param string Title text
    * @param array (optional) CSS styles
    *
    * @return JsSlideshow
    */
    public function addBullet($text, $style = null)
    {
        $slide = $this->get($this->currentSlideIndex);
        $item = array();
        $item['type'] = 'bullet';
        $item['html'] = '<p class="bullet">'.$this->prepareTabs(self::parseTags(str_replace($this->replaces['from'], $this->replaces['to'], $text))).'</p>';
        $item['style'] = isset($slide->style['fontSize']) ? array('fontSize'=>$slide->style['fontSize']) : array();
        if (isset($style) && is_array($style)) {
            $item['style'] = array_merge($item['style'], $style);
        }
        $slide->items[] = $item;
        return $this;
    }


    /**
    * Adds a Javascript code to the slideshow
    *
    * @example $slideshow->addJs( 'function x2(val){ return val*2; }' );
    *
    * @access public
    *
    * @param string Javascript code
    *
    * @return JsSlideshow
    */
    public function addJs($code)
    {
        $this->slideshowScripts[] = ";".$code;
        return $this;
    }


    /**
    * Adds a Javascript file to the slideshow
    *
    * @example $slideshow->addJsFile( 'path/to/javascript/file.js' );
    *
    * @access public
    *
    * @param string Path to the Javascript file
    *
    * @return JsSlideshow
    */
    public function addJsFile($path)
    {
        if (!file_exists($path)) {
            throw new Exception("Script file not found");
        }

        $this->slideshowScripts[] = ";".file_get_contents($path);
        return $this;
    }


    /**
    * Adds an image file to the slideshow
    *
    * @example $slideshow->addImage( 'path/to/the/image.jpg' );
    * @example $slideshow->addImage( 'path/to/the/image.jpg',
    *                               [ 'border' => 'solid 2 px red' ] );
    *
    * @access public
    *
    * @param string Path to the image file
    * @param array (optional) CSS styles
    *
    * @return JsSlideshow
    */
    public function addImage($path, $style = null)
    {
        $slide = $this->get($this->currentSlideIndex);
        $item = array();
        $image_type = 'jpeg';
        if(strpos(strtolower($path), 'jpg') !== true) {
	        $image_type = 'png';
        }
        $item['type'] = 'image';
        $this->gcsInsert($path, 'image/'.$image_type);
        unlink($path);
        $path = $this->gcs['url'].$path;
        $item['src'] = $path;
        
        $item['html'] = '<p class="image"><img src="'.$path.'" /></p>';
        $item['style'] = isset($slide->style['fontSize']) ? array('fontSize'=>$slide->style['fontSize']) : array();
        if (isset($style) && is_array($style)) {
            $item['style'] = array_merge($item['style'], $style);
        }
        $slide->items[] = $item;
        return $this;
    }

    /**
    * Adds centered text on a slide
    *
    * @example $slideshow->addCenteredText( 'Centered text' );
    * @example $slideshow->addCenteredText( 'Centered text', [ 'text-shadow' => ' 4px 4px 2px #000' ] ); //With CSS
    *
    * @access public
    *
    * @param string Text
    * @param array (optional) CSS styles
    *
    * @return JsSlideshow
    */
    public function addCenteredText($text, $style = null)
    {
        $slide = $this->get($this->currentSlideIndex);
        $item = array();
        $item['type'] = 'centered';
        $item['html'] = '<p class="centered">'.self::parseTags(str_replace($this->replaces['from'], $this->replaces['to'], $text)).'</p>';
        $item['style'] = isset($slide->style['fontSize']) ? array('fontSize'=>$slide->style['fontSize']) : array();
        if (isset($style) && is_array($style)) {
            $item['style'] = array_merge($item['style'], $style);
        }
        $slide->items[] = $item;
        return $this;
    }


    /**
    * Adds a set of centered text in the slideshow. Each text automatically creates a new slide
    *
    * @example $slideshow->addCenteredTextSet(
    *            [5.000, 'Centered text'],
    *            ... ,
    *            [null, 'Centered text', ['white-space' => 'nowrap']]
    *           );
    *
    * @access public
    *
    * @param array,... unlimited arrays in format array(time, text[, style])
    *
    * @return JsSlideshow
    */
    public function addCenteredTextSet()
    {
        $items = func_get_args();
        foreach ($items as $item) {
            $this->clear();
            $this->addSlide($item[0]);
            $style = (isset($item[2]) && is_array($item[2])) ? $item[2] : null;
            $this->addCenteredText($item[1], $style);
        }
        return $this;
    }

    /**
    * Adds a set of bullets in the slide. Each bullet placed to the same slide
    *
    * @example $slideshow->addSlideBullets(
    *             [2.000, 'First bullet\t text'],
    *             ... ,
    *             [ null, 'Last bullet:\t text', ['color' => '#cc00aa']]
    *          );
    *
    * @access public
    *
    * @param array,... unlimited arrays in format array(time, text[, style])
    *
    * @return JsSlideshow
    */
    public function addSlideBullets()
    {
        $items = func_get_args();
        foreach ($items as $item) {
            $this->addSlide($item[0]);
            $this->addBullet($item[1]);
        }
        return $this;
    }

    /**
    * Sets the clearSlide flag to true, which means that the next slide will be cleared
    *
    * @access public
    *
    * @return JsSlideshow
    */
    public function clear()
    {
        $this->clearSlide = true;
        return $this;
    }



    /**
    * Sets the slide font size
    *
    * @access public
    *
    * @param ineger Font size
    * @param string (optional) Font size units, default - px
    *
    * @return JsSlideshow
    */
    public function setFontSize($size, $units = null)
    {
        $slide = $this->get($this->currentSlideIndex);
        $slide->css["fontSize"] = $size.($units ? $units : 'px');
        return $this;
    }


    /**
    * Adds a audio file to the slideshow
    *
    * @access public
    *
    * @param string Path to the audio file
    *
    * @return JsSlideshow
    */
    public function addAudio($path)
    {
        if (!file_exists($path)) {
            throw new Exception("Audio file not found");
        }
            
        $duration = self::mp3FileLength($path);
        $this->slideshowSounds[] = array(
            'filename'=>$path,
            'duration'=>$duration
        );
        $this->slideshowPlaytime += $duration;
        return $this;
    }


    /**
    * Generates an Birth Tiles image and add its to slide
    *
    * @access public
    *
    * @param integer Year of birth
    * @param integer Month of birth
    * @param integer Day of birth
    * @param array (optional) CSS styles
    *
    * @return JsSlideshow
    */
    public function addBirthTiles($year, $m, $day, $style = null)
    {

        $months = array(
            "", "January", "February", "March", "April", "May", "June",
            "July", "August", "September", "October", "November", "December");
        $month = $months[$m];
        $yyyy=0;
        $mm=0;
        $dd=0;
        for ($i=0; $i<4; $i++) {
            $yyyy += (int)substr($year, $i, 1);
        }

        $canvas = imagecreatetruecolor($this->slideshowStyles->width-40, 225);
        imagealphablending($canvas, false);
        $transparency = imagecolorallocatealpha($canvas, 0, 0, 0, 127);
        imagefill($canvas, 0, 0, $transparency);
        imagesavealpha($canvas, true);
        $canvasWidth = imagesx($canvas);
        $canvasHeight = imagesy($canvas);


        $yy = (strlen($yyyy)>1 && $yyyy!==11 && $yyyy!==22) ? (substr($yyyy, 0, 1)+substr($yyyy, 1, 1)) : $yyyy;
        $mm = (strlen($m)>1 && $m!=11) ? (substr($m, 0, 1)+substr($m, 1, 1)) : $m;
        $dd = (strlen($day)>1 && $day!=11 && $day!=22) ? (substr($day, 0, 1)+substr($day, 1, 1)) : $day;
        $dd = ($dd==10) ? 1 : $dd;
        $ymd = $yy+$mm+$dd;
        $offsetX = (strlen($ymd)>1 && $ymd!=11 && $ymd!=22) ? (($canvasWidth/2)-294) : (($canvasWidth/2)-241);



        $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/start-column/".$year.".png", $offsetX, $canvasHeight - 68);
        $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/start-column/".intval($day).".png", $offsetX, $canvasHeight - 136);
        $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/start-column/".$month.".png", $offsetX, $canvasHeight - 204);
            
        $yearColx2 = 0;
        
        for ($i=0; $i<4; $i++) {
            $yearColx2 += (int)substr($year, $i, 1);
        }

        if (strlen($yearColx2)>1) {
            $yearColx3 = 0;
            
            if ($yearColx2==11 || $yearColx2==22) {
                $yearColx3 = $yearColx2;
            } else {
                for ($i=0; $i<2; $i++) {
                    $yearColx3 += (int)substr($yearColx2, $i, 1);
                }
            }
                
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow.png", $offsetX+164, $canvasHeight - 68);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/middle-column/".$yearColx2.".png", $offsetX+222, $canvasHeight - 68);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow.png", $offsetX+270, $canvasHeight - 68);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/middle-column/".$yearColx3.".png", $offsetX+328, $canvasHeight - 68);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrowu.png", $offsetX+376, $canvasHeight - 78);
        } else {
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow2x.png", $offsetX+164, $canvasHeight - 68);
            $yearColx3 =$yearColx2;
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/middle-column/".$yearColx3.".png", $offsetX+328, $canvasHeight - 68);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrowu.png", $offsetX+376, $canvasHeight - 78);
        }

        $monthColx3 = 0;
     
        if (strlen($m)>1 && $m!=11) {
            $monthColx3 = substr($m, 0, 1) + substr($m, 1, 1);
            
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow.png", $offsetX+164, $canvasHeight - 204);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/middle-column/".intval($m).".png", $offsetX+222, $canvasHeight - 204);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow.png", $offsetX+270, $canvasHeight - 204);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/middle-column/".intval($monthColx3).".png", $offsetX+328, $canvasHeight - 204);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrowd.png", $offsetX+376, $canvasHeight - 194);
        } else {
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow2x.png", $offsetX+164, $canvasHeight - 204);
            $monthColx3 = $m;
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/middle-column/".intval($monthColx3).".png", $offsetX+328, $canvasHeight - 204);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrowd.png", $offsetX+376, $canvasHeight - 194);
        }

        $dayColx2 = 0;
        $dayColx3 = 0;
        if (strlen($day)>1 && $day!=11 && $day!=22) {
            $dayColx2 = substr($day, 0, 1) + substr($day, 1, 1);
            ;
            
            if (strlen($dayColx2)>1) {
                $dayColx3 = substr($dayColx2, 0, 1) + substr($dayColx2, 1, 1);

                $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow.png", $offsetX+164, $canvasHeight - 136);
                $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/middle-column/".intval($dayColx2).".png", $offsetX+222, $canvasHeight - 136);
                $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow.png", $offsetX+270, $canvasHeight - 136);
                $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/middle-column/".intval($dayColx3).".png", $offsetX+328, $canvasHeight - 136);
                $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow.png", $offsetX+376, $canvasHeight - 136);
            } else {
                $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow2x.png", $offsetX+164, $canvasHeight - 136);
                $dayColx3 = $dayColx2;
                $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/middle-column/".intval($dayColx3).".png", $offsetX+328, $canvasHeight - 136);
                $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow.png", $offsetX+376, $canvasHeight - 136);
            }
            
            
        } else {
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow2x.png", $offsetX+164, $canvasHeight - 136);
            $dayColx3 = $day;
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/middle-column/".intval($dayColx3).".png", $offsetX+328, $canvasHeight - 136);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow.png", $offsetX+376, $canvasHeight - 136);
        }

        $penultCol = $yearColx3+$monthColx3+$dayColx3;
        $lastCol = 0;

        if (strlen($penultCol)>1 && $penultCol!==11 && $penultCol!==22) {
            for ($i=0; $i<2; $i++) {
                $lastCol += (int)substr($penultCol, $i, 1);
            }
        
            $lastCol = (strlen($lastCol)>1) ? substr($lastCol, 0, 1)+substr($lastCol, 1, 1) : $lastCol;

            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/penult-column/".$penultCol.".png", $offsetX+434, $canvasHeight - 136);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/arrow.png", $offsetX+482, $canvasHeight - 136);
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/last-column/".$lastCol.".png", $offsetX+540, $canvasHeight - 136);
        } else {
            $this->drawImageOnCanvas($canvas, $this->tilesImagesDirectory."/numbers/penult-column/".$penultCol.".png", $offsetX+434, $canvasHeight - 136);
        }

        $output = implode('/', array($this->slidesOutputPath, $this->slideshowID, md5('birthtiles-'.$year.$m.$day).'.png'));
        if (!file_exists(dirname($output))) {
            mkdir(dirname($output), 0777, true);
        }
        imagepng($canvas, $output, -1);
        imagedestroy($canvas);

        return $this->addImage($output, $style);
    }
        

    /**
    * Generates an Name Tiles image and add its to slide
    *
    * @access public
    *
    * @param string Full name
    * @param array (optional) CSS styles
    *
    * @return JsSlideshow
    */
    public function addTiles($text, $style = null)
    {
        $leftOffset = 0;
        
        $vowels = "aeiouyAEIOUY";
        
        $text = self::normalizeText($text);
        $tilearr = str_split($text);
        
        $tile = array();
        foreach ($tilearr as $k => $char) {
            $digit = self::toDigit($char);
            $tile[$k]['char'] =  $char;
            $tile[$k]['digit'] = $digit;
            $tile[$k]['charCode'] = self::ordutf8($char);
            $tile[$k]['digitCode'] = self::ordutf8($digit);
        }

        $letters = count($tile);
        $maxLetterWidth = floor((($this->slideshowStyles->width-40) - $letters)/$letters);

        if ($maxLetterWidth>=34) {
            $charWidth = 34;
            $charHeight = 34;
        } elseif ($maxLetterWidth<=34 && $maxLetterWidth>=24) {
            $charWidth = 24;
            $charHeight = 24;
        } elseif ($maxLetterWidth<24) {
            $charWidth = 18;
            $charHeight = 18;
        }

        $topOffset = $charHeight+5;

        $canvas = imagecreatetruecolor($this->slideshowStyles->width-40, ($charHeight*3)+10);
        imagealphablending($canvas, false);
        $transparency = imagecolorallocatealpha($canvas, 0, 0, 0, 127);
        imagefill($canvas, 0, 0, $transparency);
        imagesavealpha($canvas, true);
        $canvasWidth = imagesx($canvas);
        $canvasHeight = imagesy($canvas);
        

        foreach ($tile as $key => $char) {
            $letter = imagecreatefrompng($this->tilesImagesDirectory."/abc/".$charWidth."/".$char['charCode'].".png");
            $digit = imagecreatefrompng($this->tilesImagesDirectory."/abc/".$charWidth."/".$char['digitCode'].".png");

            $digitOffset = (preg_match("/[aeiouyAEIOUY]/", $char['char'])) ? $topOffset-$charHeight : $topOffset+$charHeight;
            $left = ($canvasWidth/2)-(($letters*$charWidth)+$letters)/2;

            imagecopy($canvas, $letter, $left+$leftOffset, $topOffset, 0, 0, $charWidth, $charHeight);
            imagecopy($canvas, $digit, $left+$leftOffset, $digitOffset, 0, 0, $charWidth, $charHeight);
            $leftOffset += $charWidth+1;
        }

        $output = implode('/', array($this->slidesOutputPath, $this->slideshowID, md5('tiles-'.$text).'.png'));
        if (!file_exists(dirname($output))) {
            mkdir(dirname($output), 0777, true);
        }
        imagepng($canvas, $output, -1);
        imagedestroy($canvas);
        return $this->addImage($output, $style);
    }


    /**
    * Compile added slides to JSON scenario, joins audio, joins script and save data to slides directory
    *
    * @example <br />
    *  $slideshow = new JsSlideshow();<br />
    *  $slideshow->addAudio($path);<br />
    *  $slideshow->addSlide($time);<br />
    *  $slideshow->setTitle($title);<br />
    *  $slideshowpath = $slideshow->compile();<br />
    *
    * @access public
    *
    * @return string Path to the compiled slideshow
    */
    public function compile($encode=False)
    {
        $sound = $this->joinAudio($encode);
        if ($sound === false) {
            throw new Exception("Unable to join audio");
        }

        $slideshow = new stdClass();
        $slideshow->audio = $this->gcs['url'].$sound;

        $script = $this->joinScripts();
        if ($script !== false) {
            $slideshow->script = $this->gcs['url'].$script;
        }

        $slideshow->playtime = $this->slideshowPlaytime;
        $slideshow->slides = $this->slides;
        $slideshow->id = $this->slideshowID;
        $slideshow->style = $this->processStyle($this->slideshowStyles);
        file_put_contents(dirname($sound).'/slideshow.json', json_encode($slideshow));
        
		$this->gcsInsert(dirname($sound).'/slideshow.json', "application/json");          
        unlink(dirname($sound).'/slideshow.json');
        $sound_file = dirname($sound).'/sound.mp3';
        if (file_exists($sound_file)) {
            unlink($sound_file);
        }
        rmdir(dirname($sound));
        return $this->gcs['url'].dirname($sound).'/slideshow.json';
    }


    /**
    * Gets the duration of mp3 file
    *
    * @access public
    * @static
    *
    * @param string Path to the audio file
    *
    * @return float Duration in seconds
    */
    public static function mp3FileLength($path)
    {
        $ID3 = new getID3();
        $info = @$ID3->analyze($path);
        $duration = floatval($info["playtime_seconds"]);
        return $duration;
    }


    /**
    * Replace scenario tags with HTML tags
    *
    * @access public
    * @static
    *
    * @param string Text to parse
    *
    * @return string Result text
    */
    public static function parseTags($text)
    {
        $text = str_replace(
            array("<red>", "<b>", "<i>", "</red>", "</b>", "</i>", "\n"),
            array("<span class=\"red\">", "<strong>", "<em>", "</span>", "</strong>", "</em>", "<br>"),
            $text
        );
        return $text;
    }


    /**
    * Generate unique id
    *
    * @access public
    * @static
    *
    * @return string Unique id
    */
    public static function uniqID()
    {
        $salt = time();
        return md5(session_id().rand()).md5($salt.rand());
    }


    /**
    * Joins all slideshow scripts to one file and save it into slideshow directory
    *
    * @access private
    *
    * @return string|false Path to the combined js file or FALSE if no scripts added
    */
    private function joinScripts()
    {
        if (count($this->slideshowScripts)) {
            $script = implode("\n;", $this->slideshowScripts);
            if (class_exists('Minifier')) {
                 $script = Minifier::minify($script);
            }
            $output = implode('/', array($this->slidesOutputPath, $this->slideshowID, self::uniqID().'.js'));

            file_put_contents($output, $script);
            
			$this->gcsInsert($output, "text/javascript");            
            unlink($output);
            return $output;
        }
        return false;
    }

    /**
    * Convert slideshow style params to css styles.
    *
    * @access private
    *
    * @param stdClass Styles object
    *
    * @return array CSS styles associative array
    */
    private function processStyle($styles)
    {
        $_styles = get_object_vars($styles);
        foreach ($_styles as $name => &$style) {
            switch ($name) {
                case 'background':
                    $style = 'url('.$style.')';
                    break;
                case 'width':
                case 'height':
                    $style = $style.'px';
                    break;
                default:
                    break;
            }
        }
        return $_styles;
    }

    /**
    * Joins all slideshow audio to one file and save it into slideshow directory.
    * That method used ffmpeg to concatenate audio.
    *
    * @access private
    *
    * @return string|boolean Path to the combined audio file or FALSE if no scripts added
    */
    private function joinAudio($encode=false)
    {
        if (count($this->slideshowSounds)) {
            $ret = 1;
            $output = implode('/', array($this->slidesOutputPath, $this->slideshowID, 'sound.mp3'));
            $sounds = array();
            foreach ($this->slideshowSounds as $sound) {
                $sounds[] = $sound['filename'];
            }

            if (!file_exists(dirname($output))) {
                mkdir(dirname($output), 0777, true);
            }
			
            if (count($this->slideshowSounds) > 1) {
                if ($encode) {
                    $cmd = '-i concat:"'.implode('|', $sounds).'" -map_metadata -1 -vn -c:a mp3 -q:a 2 -f mp3 '.$output.' -y';
                } else {
                    $cmd = '-i "concat:'.implode('|', $sounds).'" -map_metadata -1 -vn -c:a copy -f mp3 '.$output.' -y';
                }
            } else {
                if ($encode) {
                    $cmd = '-i '.$sounds[0].' -vn -c:a mp3 -q:a 2 -f mp3 '.$output.' -y';
                } else {
                    $cmd = '-i '.$sounds[0].' -vn -c:a copy -f mp3 '.$output.' -y';
                }
            }
            
            /* Modification by Nick - Send audio path to GCE server for processing and then download it. */
            
            $cmd = urlencode($cmd);
            $ch = curl_init();  
            curl_setopt($ch, CURLOPT_VERBOSE, true);
            curl_setopt($ch,CURLOPT_URL, 'http://gce-ffmpeg-server.numerologist.com/mesl/process-video.php');
            curl_setopt($ch,CURLOPT_RETURNTRANSFER,true);
            curl_setopt($ch,CURLOPT_HEADER, false); 
            curl_setopt($ch,CURLOPT_POST, 1);
            curl_setopt($ch,CURLOPT_POSTFIELDS, "cmd=$cmd&output=$output&slideshowID={$this->slideshowID}");    
            $ret=curl_exec($ch);
            curl_close($ch);
            
            $download_file = fopen($output, "w");
            $handle = curl_init("http://gce-ffmpeg-server.numerologist.com/mesl/{$output}");
            curl_setopt($handle, CURLOPT_FILE, $download_file);
            curl_setopt($handle, CURLOPT_FOLLOWLOCATION, true);
            curl_exec($handle);
            curl_close($handle);
            fclose($download_file);
            $this->gcsInsert($output); 
            unlink($output);             
            return $output;
        }
        return false;
    }


    /**
    * Replaces the letters of their numerological values
    *
    * @access public
    * @static
    *
    * @param string Text to parse
    *
    * @return string Result text
    */
    public static function toDigit($text)
    {
        $text = strtolower($text);
        $search = str_split('ajsbktcludmvenwfoxgpyhqzir');
        $replace = str_split('11122233344455566677788899');
        return str_replace($search, $replace, $text);
    }


    /**
    * Replaces letters with umlauts or diactrics their ASCII analogues
    *
    * @access public
    * @static
    *
    * @param string Text to process
    *
    * @return string Result text
    */
    public static function normalizeText($string)
    {
        $string = htmlentities($string, ENT_QUOTES, 'UTF-8');

        $string = preg_replace(
            '~&([a-z]{1, 2})(acute|cedil|circ|grave|lig|orn|ring|slash|th|tilde|uml);~i',
            '$1',
            $string
        );

        $special = array('À', 'Á', 'Â', 'Ã', 'Ä', 'Å', 'Æ', 'Ç', 'È', 'É', 'Ê', 'Ë', 'Ì', 'Í', 'Î', 'Ï', 'Ð', 'Ñ',
            'Ò', 'Ó', 'Ô', 'Õ', 'Ö', 'Ø', 'Ù', 'Ú', 'Û', 'Ü', 'Ý', 'ß', 'à', 'á', 'â', 'ã', 'ä', 'å', 'æ', 'ç', 'è',
            'é', 'ê', 'ë', 'ì', 'í', 'î', 'ï', 'ñ', 'ò', 'ó', 'ô', 'õ', 'ö', 'ø', 'ù', 'ú', 'û', 'ü', 'ý', 'ÿ', 'Ā',
            'ā', 'Ă', 'ă', 'Ą', 'ą', 'Ć', 'ć', 'Ĉ', 'ĉ', 'Ċ', 'ċ', 'Č', 'č', 'Ď', 'ď', 'Đ', 'đ', 'Ē', 'ē', 'Ĕ', 'ĕ',
            'Ė', 'ė', 'Ę', 'ę', 'Ě', 'ě', 'Ĝ', 'ĝ', 'Ğ', 'ğ', 'Ġ', 'ġ', 'Ģ', 'ģ', 'Ĥ', 'ĥ', 'Ħ', 'ħ', 'Ĩ', 'ĩ', 'Ī',
            'ī', 'Ĭ', 'ĭ', 'Į', 'į', 'İ', 'ı', 'Ĳ', 'ĳ', 'Ĵ', 'ĵ', 'Ķ', 'ķ', 'Ĺ', 'ĺ', 'Ļ', 'ļ', 'Ľ', 'ľ', 'Ŀ', 'ŀ',
            'Ł', 'ł', 'Ń', 'ń', 'Ņ', 'ņ', 'Ň', 'ň', 'ŉ', 'Ō', 'ō', 'Ŏ', 'ŏ', 'Ő', 'ő', 'Œ', 'œ', 'Ŕ', 'ŕ', 'Ŗ', 'ŗ',
            'Ř', 'ř', 'Ś', 'ś', 'Ŝ', 'ŝ', 'Ş', 'ş', 'Š', 'š', 'Ţ', 'ţ', 'Ť', 'ť', 'Ŧ', 'ŧ', 'Ũ', 'ũ', 'Ū', 'ū', 'Ŭ',
            'ŭ', 'Ů', 'ů', 'Ű', 'ű', 'Ų', 'ų', 'Ŵ', 'ŵ', 'Ŷ', 'ŷ', 'Ÿ', 'Ź', 'ź', 'Ż', 'ż', 'Ž', 'ž', 'ſ', 'ƒ', 'Ơ',
            'ơ', 'Ư', 'ư', 'Ǎ', 'ǎ', 'Ǐ', 'ǐ', 'Ǒ', 'ǒ', 'Ǔ', 'ǔ', 'Ǖ', 'ǖ', 'Ǘ', 'ǘ', 'Ǚ', 'ǚ', 'Ǜ', 'ǜ', 'Ǻ', 'ǻ',
            'Ǽ', 'ǽ', 'Ǿ', 'ǿ');
 
        $normal = array(
            'A', 'A', 'A', 'A', 'A', 'A', 'AE', 'C', 'E', 'E', 'E', 'E', 'I', 'I', 'I', 'I', 'D', 'N', 'O', 'O', 'O',
            'O', 'O', 'O', 'U', 'U', 'U', 'U', 'Y', 's', 'a', 'a', 'a', 'a', 'a', 'a', 'ae', 'c', 'e', 'e', 'e', 'e',
            'i', 'i', 'i', 'i', 'n', 'o', 'o', 'o', 'o', 'o', 'o', 'u', 'u', 'u', 'u', 'y', 'y', 'A', 'a', 'A', 'a',
            'A', 'a', 'C', 'c', 'C', 'c', 'C', 'c', 'C', 'c', 'D', 'd', 'D', 'd', 'E', 'e', 'E', 'e', 'E', 'e', 'E',
            'e', 'E', 'e', 'G', 'g', 'G', 'g', 'G', 'g', 'G', 'g', 'H', 'h', 'H', 'h', 'I', 'i', 'I', 'i', 'I', 'i',
            'I', 'i', 'I', 'i', 'IJ', 'ij', 'J', 'j', 'K', 'k', 'L', 'l', 'L', 'l', 'L', 'l', 'L', 'l', 'l', 'l', 'N',
            'n', 'N', 'n', 'N', 'n', 'n', 'O', 'o', 'O', 'o', 'O', 'o', 'OE', 'oe', 'R', 'r', 'R', 'r', 'R', 'r', 'S',
            's', 'S', 's', 'S', 's', 'S', 's', 'T', 't', 'T', 't', 'T', 't', 'U', 'u', 'U', 'u', 'U', 'u', 'U', 'u',
            'U', 'u', 'U', 'u', 'W', 'w', 'Y', 'y', 'Y', 'Z', 'z', 'Z', 'z', 'Z', 'z', 's', 'f', 'O', 'o', 'U', 'u',
            'A', 'a', 'I', 'i', 'O', 'o', 'U', 'u', 'U', 'u', 'U', 'u', 'U', 'u', 'U', 'u', 'A', 'a', 'AE', 'ae', 'O', 'o');

        return trim(str_replace($special, $normal, $string));
    }

    /**
    * Returns the char code of the UTF-8 character
    * As ord() doesn't work with utf-8, the following function will work well
    *
    * @access public
    * @static
    *
    * @param string String with character
    * @param integer (optional) Reference, as it is not easy to split a utf-8 char-by-char. Useful to iterate on a string
    *
    * @return integer Char code
    */
    public static function ordutf8($string, $offset = 0)
    {
        $code = ord(substr($string, $offset, 1));
        if ($code >= 128) {        //otherwise 0xxxxxxx
            if ($code < 224) {
                $bytesnumber = 2;                //110xxxxx
            } elseif ($code < 240) {
                $bytesnumber = 3;        //1110xxxx
            } elseif ($code < 248) {
                $bytesnumber = 4;    //11110xxx
            }
            $codetemp = $code - 192 - ($bytesnumber > 2 ? 32 : 0) - ($bytesnumber > 3 ? 16 : 0);
            for ($i = 2; $i <= $bytesnumber; $i++) {
                $offset ++;
                $code2 = ord(substr($string, $offset, 1)) - 128;        //10xxxxxx
                $codetemp = $codetemp*64 + $code2;
            }
            $code = $codetemp;
        }
        $offset += 1;
        if ($offset >= strlen($string)) {
            $offset = -1;
        }
        return $code;
    }


    /**
    * Create GD image from GIF, JPEG or PNG file
    *
    * @access private
    *
    * @param string Filename of image file
    * @param integer (optional) GD constant with image type. Supported constants: IMAGETYPE_GIF, IMAGETYPE_JPEG, IMAGETYPE_PNG. @see: http://php.net/manual/ru/image.constants.php
    *
    * @return resource|boolean Returns an image resource identifier on success, FALSE on errors.
    */
    private function imagecreatefrom($file, $type = null)
    {
        if (!$type) {
            list($width, $height, $type) = getimagesize($file);
        }
        switch ($type)
        {
            case IMAGETYPE_GIF:
                return imagecreatefromgif($file);
                break;
            case IMAGETYPE_JPEG:
                return imagecreatefromjpeg($file);
                break;
            case IMAGETYPE_PNG:
                return imagecreatefrompng($file);
                break;
            default:
                throw new Exception('This image type not supported. Currently supported only GIF, JPEG and PNG files.');
                break;
        }
    }

    /**
    * Split string with TAB characters
    *
    * @access private
    *
    * @param string Text to split
    *
    * @return string Result text
    */
    private function prepareTabs($text)
    {
        if (strpos($text, "\t") !== false) {
            $parts = preg_split('/\t/', $text, 2);
            return '<span  class="left" style="width:'.$this->tabWidth.'%">'.
                    $parts[0].'</span><span class="right" style="width:'.
                    (100-$this->tabWidth).'%">'.$parts[1].'</span>';
        }
        return $text;
    }

    public function addReplace($from,$to){
        $this->replaces['from'][] = $from;
        $this->replaces['to'][] = $to;
    }

    /**
    * Places image file to GD image
    *
    * @access private
    *
    * @param resource &$canvas GD image to draw $file on it
    * @param string Path to the image to be placed on the $canvas
    * @param int|string The position on the X axis. If $x equals "center" - $file be placed on horisontally center of $canvas
    * @param int|string The position on the Y axis. If $y equals "center" - $file be placed on vertically center of $canvas
    *
    * @return void
    */
    private function drawImageOnCanvas(&$canvas, $file, $x = "center", $y = "center")
    {
        if (is_readable($file)) {
            $canvasWidth = imagesx($canvas);
            $canvasHeight = imagesy($canvas);

            list($width, $height, $type) = getimagesize($file);
            $image = $this->imagecreatefrom($file, $type);
            if ($image) {
                if ($width > $canvasWidth || $height > $canvasHeight) {
                    $aspect = ceil($width / $height);
                    if ($width > $height) {
                        $resized = imagecreatetruecolor($canvasWidth, ceil($canvasWidth/$aspect));
                    } else {
                        $resized = imagecreatetruecolor(ceil($canvasHeight*$aspect), $canvasHeight);
                    }

                    $resizedWidth = imagesx($resized);
                    $resizedHeight = imagesy($resized);
                    imagecopyresampled($resized, $image, 0, 0, 0, 0, $resizedWidth, $resizedHeight, $canvasWidth, $canvasHeight);
                    $image = $resized;
                    $width = $resizedWidth;
                    $height = $resizedHeight;
                }

                if ($x == "center") {
                    $x = ceil(($canvasWidth-$width)/2);
                }

                if ($y == "center") {
                    $y = ceil(($canvasHeight-$height)/2);
                }

                imagecopy($canvas, $image, (int)$x, (int)$y, 0, 0, $width, $height);
            } else {
                throw new Exception("Error processing image");
            }
        } else {
            throw new Exception("Error loading file");
        }
    }
}
