<?php 
/**
* class SimpleSlide
*
* Used for simple .png slides creation
*
* @version 0.1
* @author Aleksey Yatskiv <sk65cool@gmail.com>
*
* @example example.php How to use this class
*/
class SimpleSlide {
    /** 
    * @const FONT_NORMAL Normal font style
    */
    const FONT_NORMAL       = "regular";
    /** 
    * @const FONT_BOLD Bold font style
    */
    const FONT_BOLD         = "bold";
    /** 
    * @const FONT_ITALIC Italic font style
    */
    const FONT_ITALIC       = "italic";
    /** 
    * @const FONT_BOLDITALIC Bolditalic font style
    */
    const FONT_BOLDITALIC   = "bolditalic";

    /**
    * Directory for saving outputed slides
    * @access public
    * @var string
    */
    public $slidesDir = ".";


    public $tilesImagesDir = "images";
    /**
    * Default font size in points
    * @access public
    * @var integers
    */
    public $fontSize        = 24;
    
    /**
    * Default font`s directory
    * @access public
    * @var string
    */
    public $fontsDirectory  = "fonts";
    
    /**
    * Additional indent for letter spaces
    * @access public
    * @var integer
    */
    public $spaceExtraWidth = 2;
    
    /**
    * Default top margin for slide title
    * @access public
    * @var integer
    */
    public $defaultTitleMarginTop   = 20;
    
    /**
    * Default left margin for slide title
    * @access public
    * @var integer
    */
    public $defaultTitleMarginLeft  = 20;
    
    /**
    * Default right margin for slide title
    * @access public
    * @var integer
    */
    public $defaultTitleMarginRight = 20;
    
    /**
    * Default text align fo slide`s title
    * @access public
    * @var string
    */
    public $defaultTitleTextAlign   = "center";

    /**
    * Default top margin for slide bullet
    * @access public
    * @var integer
    */
    public $defaultBulletMarginTop      = 5;

    /**
    * Default left margin for slide bullet
    * @access public
    * @var integer
    */
    public $defaultBulletMarginLeft     = 20;

    /**
    * Default right margin for slide bullet
    * @access public
    * @var integer
    */
    public $defaultBulletMarginRight    = 20;    

    /**
    * First symbol in bullet`s text line
    * @access public
    * @var string
    */
    public $bulletSymbol            = "&bull;";
    
    /**
    * Additional space between symbol and
    * first bullet`s word
    * @access public
    * @var integer
    */
    public $bulletSymbolExtraSpace  = 0;
    
    /**
    * Bullet`s first symbol font style
    * @access public
    * @var string
    */
    public $bulletSymbolFontStyle   = SimpleSlide::FONT_BOLD;

    /**
    * Default tab width in percent
    * from slide`s width 
    * @access public
    * @var integer
    */
    public $tabWidth = 40;

    /**
    * Default font styles array
    * @access public
    * @var array
    */
    public $fonts = array(
        SimpleSlide::FONT_NORMAL        => 'regular.ttf',
        SimpleSlide::FONT_BOLD          => 'bold.ttf',
        SimpleSlide::FONT_ITALIC        => 'italic.ttf',
        SimpleSlide::FONT_BOLDITALIC    => 'bolditalic.ttf',
    );

    /**
    * Background image
    * @access protected
    * @var resource
    */
    protected $background;

    /**
    * Outputed slide width in px
    * @access protected
    * @var integer
    */
    protected $slideWidth;
    
    /**
    * Outputed slide height in px
    * @access protected
    * @var integer
    */
    protected $slideHeight;
    
    /**
    * Text color, returned by imagecolorallocate
    * @access protected
    * @var recource
    */
    protected $textColor;
    
    /**
    * New line position, such as "cursor position"
    * @access protected
    * @var integer
    */
    protected $posY;


    private $_images = array();


    /**
    * Class constractor
    *
    * @param string $backgroundFile background file pathname
    * @param integer $width outputed slide width in px
    * @param integer $height outputed slide width in px
    *
    * @return void 
    */
    function __construct($backgroundFile, $width=NULL, $height=NULL) {

        if(!file_exists($backgroundFile)){
            throw new Exception('Unable to open background file');
        }

        list($backgroundWidth, $backgroundHeight, $backgroundType) = getimagesize($backgroundFile);
        $this->slideWidth = $backgroundWidth;
        $this->slideHeight = $backgroundHeight;
        $this->posY = 0;

        // select the appropriate imagecreatefrom* function based on the determined
        // image type
        $this->background = @$this->_imagecreatefrom($backgroundFile,$backgroundType);
        
        /*switch ($backgroundType)
        {
            case IMAGETYPE_GIF: 
                $this->background = @imagecreatefromgif($backgroundFile); 
                break;
            case IMAGETYPE_JPEG: 
                $this->background = @imagecreatefromjpeg($backgroundFile); 
                break;
            case IMAGETYPE_PNG: 
                $this->background = @imagecreatefrompng($backgroundFile); 
                break;
            default: 
                throw new Exception('This image type not supported. Currently supported only GIF, JPEG and PNG files.');
                break;
        }*/

        //resize background for needed widht and height
        if( $width !== NULL || $height !== NULL ){
            $aspect = ceil( $backgroundWidth / $backgroundHeight );
            if( $width !== NULL && $height !== NULL ){
                $resized = imagecreatetruecolor($width, $height);
            }elseif( $width !== NULL ){
                $resized = imagecreatetruecolor($width, ceil($width/$aspect));
            }else{
                $resized = imagecreatetruecolor(ceil($height*$aspect), $height);
            }
            $resizedWidth = imagesx($resized);
            $resizedHeight = imagesy($resized);
            imagecopyresampled($resized, $this->background, 0, 0, 0, 0, $resizedWidth, $resizedHeight, $backgroundWidth, $backgroundHeight);
            $this->background = $resized;
            $this->slideWidth = $resizedWidth;
            $this->slideHeight = $resizedHeight;
        }

        //try to load config file
        $this->loadConfig();
    }


    private function _imagecreatefrom($file, $type=NULL){
        if(!$type){
            list($_width, $_height, $type) = getimagesize($file);
        }
        switch ($type)
        {
            case IMAGETYPE_GIF:
                return imagecreatefromgif($file);
                break;
            case IMAGETYPE_JPEG:
                return imagecreatefromjpeg($file);
                break;
            case IMAGETYPE_PNG:
                return imagecreatefrompng($file);
                break;
            default:
                throw new Exception('This image type not supported. Currently supported only GIF, JPEG and PNG files.');
                break;
        }
    }

    /**
    * Create object's clone
    * 
    * @return void 
    */
    public function __clone() {
        $copy = imagecreatetruecolor($this->slideWidth, $this->slideHeight);
        imagecopy($copy, $this->background, 0, 0, 0, 0, $this->slideWidth, $this->slideHeight);
        $this->background = $copy;
    }

    /**
    * Get width of the specified slide
    * 
    * @return integer
    */
    public function getWidth(){
        return $this->slideWidth;
    }

    /**
    * Get height of the specified slide
    * 
    * @return integer
    */
    public function getHeight(){
        return $this->slideHeight;
    }

    /**
    * function loadConfig
    * 
    * @return void 
    */
    private function loadConfig(){
        if(class_exists("SimpleSlideConfig")){
            $_this = $this;
            $configVariables = SimpleSlideConfig::$config;
            $thisRef = (new ReflectionObject($_this));
            $thisVariables = $thisRef->getProperties(ReflectionProperty::IS_PUBLIC);
            array_walk($thisVariables, function($item) use ($configVariables, $_this){
                $key = $item->name;

                if(isset($configVariables[$key])){
                    if( is_array($configVariables[$item->name]) && is_array($_this->$key) ){
                        $_this->$key = array_merge($_this->$key, $configVariables[$item->name]);
                    }else{
                        $_this->$key = $configVariables[$item->name];
                    }
                }
            });
        }
    }


    /**
    * Set text color
    *
    * @param string $hexCode color code in HEX
    *
    * @return void 
    */
    public function setColor($hexCode){
        $this->textColor = $this->_getColorFromHEX($hexCode, $this->background);
    }

    /**
    * Get color from HEX code
    *
    * @param string $hexCode color code in HEX
    * @param resource $image image for color allocation
    *
    * @return resource
    */
    private function _getColorFromHEX($hexCode, $image){
        if(preg_match("/^#([A-Fa-f0-9]{6})$/", trim($hexCode) )){
            list($r, $g, $b) = sscanf($hexCode, "#%2x%2x%2x");
        }else{
            throw new Exception('Color code is not valid. Use HEX color codes');
        }
        $color = imagecolorallocate($image, $r, $g, $b);

        return $color; 
    }

    /**
    * Set fonts
    *
    * @param string $filename fonts pathname 
    * @param string $type fonts style
    *
    * @return void
    */
    public function setFont($filename, $type){
        if(!isset($type)){
            throw new Exception('Font type is unknown');
        }
        if( file_exists($filename) ){
            $this->fonts[$type] = $filename;    
        }else{
            throw new Exception('Font file not found');
        }
    }


    /**
    * Set fonts size
    *
    * @param integer $size font size in px
    *
    * @return void 
    */
    public function setFontSize($size){
        $this->fontSize = $size;
    }

    /**
    * Get path for font style.
    *
    * @param string $style font`s style
    *
    * @return string pathname of the font 
    */
    private function getFont($style){
        return $this->fontsDirectory."/".$this->fonts[$style];
    }


    /**
    * Set tab size
    *
    * @param integer $persent tab width 
    * in percent from slide width
    *
    * @return void
    */
    public function setTab($percent){
        $this->tabWidth = $percent;
    }

    /**
    * Mark bold words in text
    *
    * @param string $text text part for check
    *
    * @return string
    */
    private function _markBoldWords($text){
        return preg_replace_callback(
                "/(<b>|<strong>)(\s*?)(.*)(\s*?)(<\/b>|<\/strong>)/ismU", 
                function($matches){
                    $result = preg_replace("!(\s+)!iU", "$1[*SimpleSlide::BOLD*]", trim($matches[3]));
                    return $matches[2]."[*SimpleSlide::BOLD*]".$result.$matches[4];
                }, 
                $text);
    }

    /**
    * Mark italic words in text
    *
    * @param string $text text part for check
    * 
    * @return string
    */
    private function _markItalicWords($text){
        return preg_replace_callback(
                "/(<i>|<em>)(\s*?)(.*)(\s*?)(<\/i>|<\/em>)/ismU", 
                function($matches){
                    $result = preg_replace("!\s+!iU", " [*SimpleSlide::ITALIC*]", trim($matches[3]));
                    return $matches[2]."[*SimpleSlide::ITALIC*]".$result.$matches[4];
                }, 
                $text);
    }

    /**
    * Mark line breaks in text
    *
    * @param string $text text part for check
    *
    * @return string
    */
    private function _markLineBreaks($text){
        return preg_replace("/(\n|\r\n)/ismU", "[*SimpleSlide::LINE_BREAK*] ", $text);
    }

    /**
    * Mark words colors in text
    *
    * @param string $text text part for check
    *
    * @return string
    */
    private function _markWordsColors($text){
        $text = preg_replace_callback(
                "/(<red>)(\s*?)(.*)(\s*?)(<\/red>)/ismU", 
                function($matches){
                    $result = preg_replace("!(\s+)!iU", "$1[*SimpleSlide::COLOR=#FF0000*]", trim($matches[3]));
                    return $matches[2]."[*SimpleSlide::COLOR=#FF0000*]".$result.$matches[4];
                }, 
                $text);

        $text = preg_replace_callback(
                "/(<color:(#[A-Fa-f0-9]{6})>)(\s*?)(.*)(\s*?)(<\/color>)/ismU",
                function($matches){
                    $result = preg_replace("!(\s+)!iU", "$1[*SimpleSlide::COLOR=".$matches[2]."*]", trim($matches[4]));
                    return $matches[3]."[*SimpleSlide::COLOR=".$matches[2]."*]".$result.$matches[5];
                }, 
                $text);
        return $text;
    }

    /**
    * Checks lineheight for current font size
    *
    * @param string $fontSize current font size
    *
    * @return integer
    */
    private function _getLineHeightPx($fontSize){
        //Finds line height in px for with point of "upper" and "lower" font symbols
        $bbox = imagettfbbox($fontSize, 0, $this->getFont(SimpleSlide::FONT_NORMAL), "WXYyp|^_ÜjQg"); 
        $lineHeightPx = abs($bbox[7]) - abs($bbox[1]);
        if($bbox[3] > 0) {
            $lineHeightPx = abs($bbox[7] - $bbox[1]) - 1;
        }        
        return $lineHeightPx;
    }

    /**
    * Defines word marked style
    *
    * @param string $word word for check
    * 
    * @return string
    */
    private function _determineWordFontStyle($word){
        if(strpos($word, "[*SimpleSlide::BOLD*]")!==FALSE && strpos($word, "[*SimpleSlide::ITALIC*]")!==FALSE){
            $currentFont = SimpleSlide::FONT_BOLDITALIC;
        }elseif(strpos($word, "[*SimpleSlide::BOLD*]")!==FALSE){
            $currentFont = SimpleSlide::FONT_BOLD;
        }elseif(strpos($word, "[*SimpleSlide::ITALIC*]")!==FALSE){
            $currentFont = SimpleSlide::FONT_ITALIC;
        }else{
            $currentFont = SimpleSlide::FONT_NORMAL;
        }
        
        return $currentFont;
    }

    /**
    * Adding title to the slide
    *
    * @param string $title title text
    * @param string $textAlign
    * @param integer $marginTop top margin in px
    * @param integer $marginLeft left margin in px
    * @param integer $marginRight right margin in px
    *
    * @return void
    */
    public function setTitle($title, $textAlign=NULL, $marginTop=NULL, $marginLeft=NULL, $marginRight=NULL){

        if($textAlign === NULL){
            $textAlign = $this->defaultTitleTextAlign;
        }

        if($marginTop === NULL){
            $marginTop = $this->defaultTitleMarginTop;
        }

        if($marginLeft === NULL){
            $marginLeft = $this->defaultTitleMarginLeft;
        }

        if($marginRight === NULL){
            $marginRight = $this->defaultTitleMarginRight;
        }

        $this->writeMultilineText($title,$textAlign,$this->posY+$marginTop,$marginLeft,$marginRight);
    }

    /**
    * Adding bullet text paragraph
    *
    * @param string $text bullet text
    * @param integer $marginTop top margin in px
    * @param integer $marginLeft left margin in px
    * @param integer $marginRight right margin in px
    *
    * @return void
    */
    public function addBullet($text, $marginTop=NULL, $marginLeft=NULL, $marginRight=NULL){

        if($marginTop === NULL){
            $marginTop = $this->defaultBulletMarginTop;
        }

        if($marginLeft === NULL){
            $marginLeft = $this->defaultBulletMarginLeft;
        }

        if($marginRight === NULL){
            $marginRight = $this->defaultBulletMarginRight;
        }

        imagettftext($this->background, $this->fontSize, 0, $marginLeft, $this->posY+$this->fontSize+$marginTop, $this->textColor, $this->getFont($this->bulletSymbolFontStyle), html_entity_decode($this->bulletSymbol." ", ENT_COMPAT, 'UTF-8'));
        $bbox = imagettfbbox($this->fontSize, 0, $this->getFont($this->bulletSymbolFontStyle), html_entity_decode($this->bulletSymbol." ") );

        if(strpos($text,"\t")!==FALSE){

            $parts = explode("\t", $text, 2);
            $bulletTabWidth = round(( $this->slideWidth - $marginLeft - $marginRight ) * ($this->tabWidth/100));
            $lineHeightPx = $this->_getLineHeightPx($this->fontSize);

            $linesLeft = $this->writeMultilineText(
                $parts[0],
                "left",
                $this->posY+$marginTop,
                $marginLeft+$bbox[4]+$this->bulletSymbolExtraSpace, 
                $this->slideWidth - $bulletTabWidth - $marginRight
            );

            $linesRight = $this->writeMultilineText(
                $parts[1],
                "left",
                ($this->posY-(($linesLeft)*$lineHeightPx) ),
                $marginLeft+$bulletTabWidth, 
                $marginRight
            );

            if($linesLeft>$linesRight){
                $this->posY = $this->posY + (($linesLeft-$linesRight)*$lineHeightPx);
            }
        }else{
            $this->writeMultilineText( $text,"left",$this->posY+$marginTop,$marginLeft+$bbox[4]+$this->bulletSymbolExtraSpace,$marginRight);
        }
    }
    
    /**
    * Print multiline text
    * vertically and horizontally aligned
    *
    * @param string $text text
    * @param integer $marginLeft left margin in px
    * @param integer $marginRight right margin in px
    *
    * @return integer vaoid
    */
    public function addCenteredText($text, $marginLeft=NULL, $marginRight=NULL){
        if($marginLeft === NULL){
            $marginLeft = $this->defaultTitleMarginLeft;
        }

        if($marginRight === NULL){
            $marginRight = $this->defaultTitleMarginRight;
        }
        $this->writeMultilineText( $text,"center",0,$marginLeft,$marginRight,TRUE);
    }


    /**
    * Print multiline text width auto-wrapp on slide
    *
    * @param string $text title or bullet text
    * @param string $textAlign
    * @param integer $marginTop top margin in px
    * @param integer $marginLeft left margin in px
    * @param integer $marginRight right margin in px
    * @param boolean $verticalCenter vertically centered text
    *
    * @return integer $totaLines
    */
    public function writeMultilineText($text, $textAlign="left", $topMargin=0, $leftMargin=0, $rightMargin=0, $verticalCenter=FALSE){

        $text = html_entity_decode($text);

        //Mark all break lines
        $text = $this->_markLineBreaks($text);

        //Mark all bold words
        $text = $this->_markBoldWords($text);

        //Mark all italic words
        $text = $this->_markItalicWords($text);

        //Mark words color
        $text = $this->_markWordsColors($text);

        //Subdividing text for words
        $words = explode(" ", $text);

        //Cheking line height in px
        $lineHeightPx = $this->_getLineHeightPx($this->fontSize);

        $newLine = TRUE;
        $finishLine = FALSE;
        $lineWidth = 0;
        $totalLines = 0;
        //Set normal font by default
        $currentFont = SimpleSlide::FONT_NORMAL;
        $wordsCount = count($words);

        $tmpBackground = imagecreatetruecolor($this->slideWidth, $this->slideHeight);
        imagealphablending($tmpBackground, false);
        $transparency = imagecolorallocatealpha($tmpBackground, 0, 0, 0, 127);
        imagefill($tmpBackground, 0, 0, $transparency);
        imagesavealpha($tmpBackground, true);


        for ($i=0; $i < $wordsCount ; $i++) { 

            if($newLine){
                //Create blank transparent pattern
                //where we will paint our text lines
                $lineImage = imagecreatetruecolor($this->slideWidth, $lineHeightPx*1.25);
                imagealphablending($lineImage, false);
                $transparency = imagecolorallocatealpha($lineImage, 0, 0, 0, 127);
                imagefill($lineImage, 0, 0, $transparency);
                imagesavealpha($lineImage, true);
                $newLine = FALSE;
                $lineWidth = 0;
            }

            //If it's first word in line, there is no space added
            $word = ($lineWidth==0) ? $words[$i] :  " ".$words[$i];
            
            //Defining font style for current word
            $currentFont = $this->_determineWordFontStyle($word);
            
            //line braks replacement
            $word = preg_replace("/\[\*SimpleSlide::LINE_BREAK\*\]/", "\n", $word);

            //if there is a color marked word
            if( preg_match("/\[\*SimpleSlide::COLOR=(#[A-Fa-f0-9]{6})\*\]/", $word, $colorMatches) ){
                $textColor = $this->_getColorFromHEX($colorMatches[1], $tmpBackground);
            }else{
                $textColor = $this->textColor;
            }

            //remove all specialy added tags
            $word = preg_replace("/\[\*SimpleSlide::[^\*]*\*\]/", "", $word);
            
            //check current word dimensions 
            $dim = imagettfbbox($this->fontSize, 0, $this->getFont($currentFont), $word);

            if($lineWidth == 0 || $lineWidth+$dim[4] <= $this->slideWidth-$leftMargin-$rightMargin){
                //adding word to the pattern
                imagettftext($lineImage, $this->fontSize, 0, $lineWidth-1, $this->fontSize, $textColor, $this->getFont($currentFont), $word);                //increase linewidth by addind word`s width and spaceExtraWidth
                $lineWidth += $dim[4]+$this->spaceExtraWidth;
            }else{
                //if word can`t placed in line
                $i--;
                $finishLine = TRUE;
            }

            //if finds line break, creates new line
            if(strpos($word, "\n")!==FALSE){
                $finishLine = TRUE;
                $newLine = TRUE;
            }

            if($finishLine || ($i==($wordsCount-1)) ){
                if( $i<($wordsCount-1) ){
                    $newLine = TRUE;
                }
                $finishLine = FALSE;
                
                //Margins calculating
                if($textAlign!="left"){
                    $lineMarginLeft = ( $textAlign == "center" ) ? (($this->slideWidth-$lineWidth)/2)+$leftMargin-$rightMargin : $this->slideWidth-$rightMargin-$lineWidth;
                }else{
                    $lineMarginLeft = $leftMargin;
                }

                //Copy our patern to our temp background
                imagecopy($tmpBackground, $lineImage, $lineMarginLeft, $topMargin, 0, 0, $lineWidth, $lineHeightPx);
                $topMargin+=$lineHeightPx;
                $totalLines++;
            }


        }
        $this->posY = $topMargin;
        //check if isset vertical aligment
        $posTop = ($verticalCenter) ? ceil(($this->slideHeight-($totalLines*$lineHeightPx))/2) : 0;
        imagecopy($this->background, $tmpBackground, 0, $posTop, 0, 0, $this->slideWidth, $this->slideHeight);
        imagedestroy($tmpBackground);
        return $totalLines;
    }


    public static function toDigit($text){
        $text = strtolower($text);
        $search = str_split('ajsbktcludmvenwfoxgpyhqzir');
        $replace = str_split('11122233344455566677788899');
     return str_replace($search,$replace,$text);
    }

    public static function normalize_text($string){
            $string = htmlentities($string, ENT_QUOTES, 'UTF-8');
            $string = preg_replace('~&([a-z]{1,2})(acute|cedil|circ|grave|lig|orn|ring|slash|th|tilde|uml);~i', '$1', $string);
            $special = array('À', 'Á', 'Â', 'Ã', 'Ä', 'Å', 'Æ', 'Ç', 'È', 'É', 'Ê', 'Ë', 'Ì', 'Í', 'Î', 'Ï', 'Ð', 'Ñ', 'Ò', 'Ó', 'Ô', 'Õ', 'Ö', 'Ø', 'Ù', 'Ú', 'Û', 'Ü', 'Ý', 'ß', 'à', 'á', 'â', 'ã', 'ä', 'å', 'æ', 'ç', 'è', 'é', 'ê', 'ë', 'ì', 'í', 'î', 'ï', 'ñ', 'ò', 'ó', 'ô', 'õ', 'ö', 'ø', 'ù', 'ú', 'û', 'ü', 'ý', 'ÿ', 'Ā', 'ā', 'Ă', 'ă', 'Ą', 'ą', 'Ć', 'ć', 'Ĉ', 'ĉ', 'Ċ', 'ċ', 'Č', 'č', 'Ď', 'ď', 'Đ', 'đ', 'Ē', 'ē', 'Ĕ', 'ĕ', 'Ė', 'ė', 'Ę', 'ę', 'Ě', 'ě', 'Ĝ', 'ĝ', 'Ğ', 'ğ', 'Ġ', 'ġ', 'Ģ', 'ģ', 'Ĥ', 'ĥ', 'Ħ', 'ħ', 'Ĩ', 'ĩ', 'Ī', 'ī', 'Ĭ', 'ĭ', 'Į', 'į', 'İ', 'ı', 'Ĳ', 'ĳ', 'Ĵ', 'ĵ', 'Ķ', 'ķ', 'Ĺ', 'ĺ', 'Ļ', 'ļ', 'Ľ', 'ľ', 'Ŀ', 'ŀ', 'Ł', 'ł', 'Ń', 'ń', 'Ņ', 'ņ', 'Ň', 'ň', 'ŉ', 'Ō', 'ō', 'Ŏ', 'ŏ', 'Ő', 'ő', 'Œ', 'œ', 'Ŕ', 'ŕ', 'Ŗ', 'ŗ', 'Ř', 'ř', 'Ś', 'ś', 'Ŝ', 'ŝ', 'Ş', 'ş', 'Š', 'š', 'Ţ', 'ţ', 'Ť', 'ť', 'Ŧ', 'ŧ', 'Ũ', 'ũ', 'Ū', 'ū', 'Ŭ', 'ŭ', 'Ů', 'ů', 'Ű', 'ű', 'Ų', 'ų', 'Ŵ', 'ŵ', 'Ŷ', 'ŷ', 'Ÿ', 'Ź', 'ź', 'Ż', 'ż', 'Ž', 'ž', 'ſ', 'ƒ', 'Ơ', 'ơ', 'Ư', 'ư', 'Ǎ', 'ǎ', 'Ǐ', 'ǐ', 'Ǒ', 'ǒ', 'Ǔ', 'ǔ', 'Ǖ', 'ǖ', 'Ǘ', 'ǘ', 'Ǚ', 'ǚ', 'Ǜ', 'ǜ', 'Ǻ', 'ǻ', 'Ǽ', 'ǽ', 'Ǿ', 'ǿ');
            $normal = array('A', 'A', 'A', 'A', 'A', 'A', 'AE', 'C', 'E', 'E', 'E', 'E', 'I', 'I', 'I', 'I', 'D', 'N', 'O', 'O', 'O', 'O', 'O', 'O', 'U', 'U', 'U', 'U', 'Y', 's', 'a', 'a', 'a', 'a', 'a', 'a', 'ae', 'c', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'n', 'o', 'o', 'o', 'o', 'o', 'o', 'u', 'u', 'u', 'u', 'y', 'y', 'A', 'a', 'A', 'a', 'A', 'a', 'C', 'c', 'C', 'c', 'C', 'c', 'C', 'c', 'D', 'd', 'D', 'd', 'E', 'e', 'E', 'e', 'E', 'e', 'E', 'e', 'E', 'e', 'G', 'g', 'G', 'g', 'G', 'g', 'G', 'g', 'H', 'h', 'H', 'h', 'I', 'i', 'I', 'i', 'I', 'i', 'I', 'i', 'I', 'i', 'IJ', 'ij', 'J', 'j', 'K', 'k', 'L', 'l', 'L', 'l', 'L', 'l', 'L', 'l', 'l', 'l', 'N', 'n', 'N', 'n', 'N', 'n', 'n', 'O', 'o', 'O', 'o', 'O', 'o', 'OE', 'oe', 'R', 'r', 'R', 'r', 'R', 'r', 'S', 's', 'S', 's', 'S', 's', 'S', 's', 'T', 't', 'T', 't', 'T', 't', 'U', 'u', 'U', 'u', 'U', 'u', 'U', 'u', 'U', 'u', 'U', 'u', 'W', 'w', 'Y', 'y', 'Y', 'Z', 'z', 'Z', 'z', 'Z', 'z', 's', 'f', 'O', 'o', 'U', 'u', 'A', 'a', 'I', 'i', 'O', 'o', 'U', 'u', 'U', 'u', 'U', 'u', 'U', 'u', 'U', 'u', 'A', 'a', 'AE', 'ae', 'O', 'o');
            return trim(str_replace($special, $normal, $string));
    }

    public static function ordutf8($string, $offset=0) {
    $code = ord(substr($string, $offset,1)); 
    if ($code >= 128) {        //otherwise 0xxxxxxx
        if ($code < 224) $bytesnumber = 2;                //110xxxxx
        else if ($code < 240) $bytesnumber = 3;        //1110xxxx
        else if ($code < 248) $bytesnumber = 4;    //11110xxx
        $codetemp = $code - 192 - ($bytesnumber > 2 ? 32 : 0) - ($bytesnumber > 3 ? 16 : 0);
        for ($i = 2; $i <= $bytesnumber; $i++) {
            $offset ++;
            $code2 = ord(substr($string, $offset, 1)) - 128;        //10xxxxxx
            $codetemp = $codetemp*64 + $code2;
        }
        $code = $codetemp;
    }
    $offset += 1;
    if ($offset >= strlen($string)) $offset = -1;
    return $code;
    }
    

    public function addImage($file, $x="center", $y="center", $moveCursor = TRUE){
        if(is_readable($file)){
            list($width, $height, $type) = getimagesize($file);
            $image = $this->_imagecreatefrom($file, $type);
            if($image){
                if( $width > $this->slideWidth || $height > $this->slideHeight ){
                    $aspect = ceil( $width / $height );
                    if( $width > $height ){
                        $resized = imagecreatetruecolor($this->slideWidth, ceil($this->slideWidth/$aspect));
                    }else{
                        $resized = imagecreatetruecolor(ceil($this->slideHeight*$aspect), $this->slideHeight);
                    }

                    $resizedWidth = imagesx($resized);
                    $resizedHeight = imagesy($resized);
                    imagecopyresampled($resized, $image, 0, 0, 0, 0, $resizedWidth, $resizedHeight, $this->slideWidth, $this->slideHeight);
                    $image = $resized;
                    $width = $resizedWidth;
                    $height = $resizedHeight;
                }

                if($x == "center"){
                    $x = ceil(($this->slideWidth-$width)/2);
                }

                if($y == "center"){
                    $y = ceil(($this->slideHeight-$height)/2);
                }

                imagecopy($this->background, $image, (int)$x, (int)$y, 0, 0, $width, $height);
                if($moveCursor){
                    $this->posY = $y+$height;
                }
            }else{
                throw new Exception("Error processing image");
            }
        }else{
            throw new Exception("Error loading file");
        }
    }

    public function addTiles($text, $fromTop=NULL){
        $topOffset = ($fromTop) ? ($fromTop*$this->slideHeight)/100 : 0;
        $leftOffset = 0;
        
        $vowels = "aeiouyAEIOUY";
        
        $text = self::normalize_text($text);
        $tilearr = str_split($text);
        
        $tile = array();
        foreach($tilearr as $k=>$char){
            $digit = self::toDigit($char);
            $tile[$k]['char'] =  $char;
            $tile[$k]['digit'] = $digit;
            $tile[$k]['charCode'] = self::ordutf8($char);
            $tile[$k]['digitCode'] = self::ordutf8($digit);
        }

        $letters = count($tile);
        $maxLetterWidth = floor(($this->slideWidth - $letters)/$letters);
        
        foreach($tile as $key=>$char){
            
            if($maxLetterWidth>=48){
                $letter = imagecreatefrompng($this->tilesImagesDir."/abc/48/".$char['charCode'].".png");
                $digit = imagecreatefrompng($this->tilesImagesDir."/abc/48/".$char['digitCode'].".png");
                $charWidth = 48;
                $charHeight = 48;  
            }else if($maxLetterWidth<=48 && $maxLetterWidth>=42){
                $letter = imagecreatefrompng($this->tilesImagesDir."/abc/42/".$char['charCode'].".png");
                $digit = imagecreatefrompng($this->tilesImagesDir."/abc/42/".$char['digitCode'].".png");
                $charWidth = 42;
                $charHeight = 42;
            }else if($maxLetterWidth<=42 && $maxLetterWidth>=38){
                $letter = imagecreatefrompng($this->tilesImagesDir."/abc/38/".$char['charCode'].".png");
                $digit = imagecreatefrompng($this->tilesImagesDir."/abc/38/".$char['digitCode'].".png");
                $charWidth = 38;
                $charHeight = 38;
            }else if($maxLetterWidth<=38 && $maxLetterWidth>=34){
                $letter = imagecreatefrompng($this->tilesImagesDir."/abc/34/".$char['charCode'].".png");
                $digit = imagecreatefrompng($this->tilesImagesDir."/abc/34/".$char['digitCode'].".png");
                $charWidth = 34;
                $charHeight = 34;
            }else if($maxLetterWidth<=34 && $maxLetterWidth>=24){
                $letter = imagecreatefrompng($this->tilesImagesDir."/abc/24/".$char['charCode'].".png");
                $digit = imagecreatefrompng($this->tilesImagesDir."/abc/24/".$char['digitCode'].".png");
                $charWidth = 24;
                $charHeight = 24;
            }else if($maxLetterWidth<24){
                $letter = imagecreatefrompng($this->tilesImagesDir."/abc/18/".$char['charCode'].".png");
                $digit = imagecreatefrompng($this->tilesImagesDir."/abc/18/".$char['digitCode'].".png");
                $charWidth = 18;
                $charHeight = 18;
            }   
                $digitOffset = (preg_match("/[aeiouyAEIOUY]/", $char['char'])) ? $topOffset-$charHeight : $topOffset+$charHeight;
                $left = ($this->slideWidth/2)-(($letters*$charWidth)+$letters)/2;

                imagecopy($this->background, $letter, $left+$leftOffset, $topOffset, 0, 0, $charWidth, $charHeight);
                imagecopy($this->background, $digit, $left+$leftOffset, $digitOffset, 0, 0, $charWidth, $charHeight);
                $leftOffset += $charWidth+1;    
        }

    }



    public function addBirthTiles($text){   
        
        $year = substr($text,0,4);
        $months = array("","January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December");
        $month = $months[(int)substr($text,4,2)];        
        $day = (int)substr($text,6,2);
        
        self::addImage($this->tilesImagesDir."/numbers/start-column/".$year.".png",10,$this->slideHeight - 68);
        self::addImage($this->tilesImagesDir."/numbers/start-column/".$day.".png",10,$this->slideHeight - 136);
        self::addImage($this->tilesImagesDir."/numbers/start-column/".$month.".png",10,$this->slideHeight - 204);
            
        $yearColx2 = 0;
        

        for($i=0; $i<4; $i++){
            $yearColx2 += (int)substr( $year, $i, 1 );
        }

        if (strlen($yearColx2)>1){
            $yearColx3 = 0;
            for($i=0; $i<2; $i++){
                $yearColx3 += (int)substr($yearColx2, $i, 1 );
            }
                
            self::addImage($this->tilesImagesDir."/arrow.png",174,$this->slideHeight - 68);    
            self::addImage($this->tilesImagesDir."/numbers/middle-column/".$yearColx2.".png",232,$this->slideHeight - 68);
            self::addImage($this->tilesImagesDir."/arrow.png",280,$this->slideHeight - 68);
            self::addImage($this->tilesImagesDir."/numbers/middle-column/".$yearColx3.".png",338,$this->slideHeight - 68);
            self::addImage($this->tilesImagesDir."/arrowu.png",386,$this->slideHeight - 78);
        }else{
            self::addImage($this->tilesImagesDir."/arrow2x.png",174,$this->slideHeight - 68);   
            $yearColx3 =$yearColx2;
            self::addImage($this->tilesImagesDir."/numbers/middle-column/".$yearColx3.".png",338,$this->slideHeight - 68);
            self::addImage($this->tilesImagesDir."/arrowu.png",386,$this->slideHeight - 78);
        }

        


        $monthColx2 = (int)substr($text,4,2);
        $monthColx3 = 0;

         
        if (strlen($monthColx2)>1){
            for($i=0; $i<2; $i++){
                $monthColx3 += (int)substr($monthColx2, $i, 1 );
            }
            self::addImage($this->tilesImagesDir."/arrow.png",174,$this->slideHeight - 204);  
            self::addImage($this->tilesImagesDir."/numbers/middle-column/".$monthColx2.".png",232,$this->slideHeight - 204);
            self::addImage($this->tilesImagesDir."/arrow.png",280,$this->slideHeight - 204);
            self::addImage($this->tilesImagesDir."/numbers/middle-column/".$monthColx3.".png",338,$this->slideHeight - 204);
            self::addImage($this->tilesImagesDir."/arrowd.png",386,$this->slideHeight - 194);
        }else{
            self::addImage($this->tilesImagesDir."/arrow2x.png",174,$this->slideHeight - 204);
            $monthColx3 = $monthColx2;
            self::addImage($this->tilesImagesDir."/numbers/middle-column/".$monthColx3.".png",338,$this->slideHeight - 204);
            self::addImage($this->tilesImagesDir."/arrowd.png",386,$this->slideHeight - 194);
        }

        $dayColx2 = 0;
        $dayColx3 = 0;
        if (strlen($day)>1){
            for($i=0; $i<2; $i++){
                $dayColx2 += (int)substr($day, $i, 1 );
            }
                if(strlen($dayColx2)>1){
                    for($i=0; $i<2; $i++){
                        $dayColx3 += (int)substr($dayColx2, $i, 1 );
                    }
                    self::addImage($this->tilesImagesDir."/arrow.png",174,$this->slideHeight - 136);
                    self::addImage($this->tilesImagesDir."/numbers/middle-column/".$dayColx2.".png",232,$this->slideHeight - 136);
                    self::addImage($this->tilesImagesDir."/arrow.png",280,$this->slideHeight - 136);
                    self::addImage($this->tilesImagesDir."/numbers/middle-column/".$dayColx3.".png",338,$this->slideHeight - 136);  
                    self::addImage($this->tilesImagesDir."/arrow.png",386,$this->slideHeight - 136);
                }else{
                    self::addImage($this->tilesImagesDir."/arrow2x.png",174,$this->slideHeight - 136);
                    $dayColx3 = $dayColx2;
                    self::addImage($this->tilesImagesDir."/numbers/middle-column/".$dayColx3.".png",338,$this->slideHeight - 136);
                    self::addImage($this->tilesImagesDir."/arrow.png",386,$this->slideHeight - 136);
                }
            
            
        }else{
            self::addImage($this->tilesImagesDir."/arrow2x.png",174,$this->slideHeight - 136);
            $dayColx3 = $day;
            self::addImage($this->tilesImagesDir."/numbers/middle-column/".$dayColx3.".png",338,$this->slideHeight - 136);
            self::addImage($this->tilesImagesDir."/arrow.png",386,$this->slideHeight - 136);
        }

        $penultCol = $yearColx3+$monthColx3+$dayColx3;
        $lastCol = 0;

        if(strlen($penultCol)>1 && $penultCol!==11 && $penultCol!==22){
            for($i=0; $i<2; $i++){
                $lastCol += (int)substr($penultCol, $i, 1 );
            }
            self::addImage($this->tilesImagesDir."/numbers/penult-column/".$penultCol.".png",444,$this->slideHeight - 136);
            self::addImage($this->tilesImagesDir."/arrow.png",502,$this->slideHeight - 136);
            self::addImage($this->tilesImagesDir."/numbers/last-column/".$lastCol.".png",560,$this->slideHeight - 136);
        }else{
            self::addImage($this->tilesImagesDir."/numbers/penult-column/".$penultCol.".png",444,$this->slideHeight - 136);
        }
    }
        
        
            /**"
    * Saves slide in .png image
    *
    * @param string $filename
    * @param string $index
    *
    * @return void
    */
    public function saveSlide($filename,$dir=NULL,$overwrite=FALSE){
        
        if($dir == NULL){
            $dir = $this->slidesDir;
        }

        $slideFilename = $dir."/".$filename;

        if(!file_exists($slideFilename) || $overwrite==TRUE){
                if(imagepng($this->background,$slideFilename)){
                    $this->_images[] = $slideFilename;
                    return $slideFilename;
                }else{
                    return FALSE;
                }
        }else{
            throw new Exception('File exists, select another name or set overwrite to TRUE');  
        }

    }

    public function getImages(){
        return $this->_images;
    }

} 

?>