<?php
/**
 * @author  Daniel J Holmes dan@djcentric.com
 * Copyright (c) 2014, Daniel J Holmes
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by Centric Web Estate.
 * 4. Neither the name of the Centric Web Estate nor the
 *    names of its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 5. This product cannot be used in a commercial environment without the
 *    express written permission of the copyright holder.
 *
 * THIS SOFTWARE IS PROVIDED BY DANIEL J HOLMES ''AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL DANIEL J HOLMES BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE SITE NUMEROLOGIST.COM AND ITS OWNER HAVE BEEN PROVIDED WITH EXPRESS
 * WRITTEN PERMISSION FOR THE USE OF THIS PRODUCT AND COPYING ITS CONTENTS IN
 * FULL BUT NOT IN PART. THIS PRODUCT MAY BE COPIED TO MULTIPLE LOCATIONS ON
 * THE SERVER HOSTING NUMEROLOGIST.COM AND MAY NOT BE COPIED ELSEWHERE WITHOUT
 * EXPRESS WRITTEN PERMISSION.
 */

require 'classes/calculators/NumberHelpers.php';

class ExpressionCalculator
{
  protected $_birth_name;
  protected $_current_name;
  protected $NUMBER_VALUES = array(
    'a' => 1,
    'b' => 2,
    'c' => 3,
    'd' => 4,
    'e' => 5,
    'f' => 6,
    'g' => 7,
    'h' => 8,
    'i' => 9,
    'j' => 10,
    'k' => 11,
    'l' => 12,
    'm' => 13,
    'n' => 14,
    'o' => 15,
    'p' => 16,
    'q' => 17,
    'r' => 18,
    's' => 19,
    't' => 20,
    'u' => 21,
    'v' => 22,
    'w' => 23,
    'x' => 24,
    'y' => 25,
    'z' => 26
  );

  protected $REDUCED_NUMBER_VALUES = array(
    'a' => 1,
    'b' => 2,
    'c' => 3,
    'd' => 4,
    'e' => 5,
    'f' => 6,
    'g' => 7,
    'h' => 8,
    'i' => 9,
    'j' => 1,
    'k' => 2,
    'l' => 3,
    'm' => 4,
    'n' => 5,
    'o' => 6,
    'p' => 7,
    'q' => 8,
    'r' => 9,
    's' => 1,
    't' => 2,
    'u' => 3,
    'v' => 4,
    'w' => 5,
    'x' => 6,
    'y' => 7,
    'z' => 8
  );

  protected $VOWELS = array ('a','e','i','o','u','w','y');
  protected $CONSONANTS = array('b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'x', 'z');

  protected $_values = array();

  public function __construct ($birth_name, $current_name)
  {
    if(!is_string($birth_name) || !is_string($current_name))
    {
      throw new \Exception('$birth_name and $current_name must be strings.');
    }
    // Lets fix the damn names
    // !u aka /u makes sure we parse utf8 nicely
    $birth_name = preg_replace('!\s+!u', ' ', trim($birth_name));
    $current_name = preg_replace('!\s+!u', ' ', trim($current_name));

    $this->_birth_name = strtolower($birth_name);
    $this->_current_name = strtolower($current_name);

    // Translate the names
    $rule = 'Any-Latin; NFD; [:Nonspacing Mark:] Remove; NFC; Lower;';

    $name = transliterator_transliterate($rule, $this->_birth_name);
    $this->_translated_birth_name = $name;

    $name = transliterator_transliterate($rule, $this->_current_name);
    $this->_translated_current_name = $name;

    $this->_calculate();
  }

  public function getBirthName ()
  {
    return $this->_birth_name;
  }

  public function getCurrentName ()
  {
    return $this->_current_name;
  }

  public function getTranslatedBirthName ()
  {
    return $this->_translated_birth_name;
  }

  public function getTranslatedCurrentName ()
  {
    return $this->_translated_current_name;
  }

  protected function _calculate ()
  {
    $this->_values['nickname_value'] = $this->_calculateNicknameValue();
    $full_name_values = $this->_calculateNameValues();
    $this->_values = array_merge($this->_values, $full_name_values);
    $this->_values['soulurge'] = $this->_calculateSoulUrge();
    $this->_values['soulurge_challenge'] = $this->_calculateSoulUrgeChallenge();
    $this->_values['personality'] = $this->_calculatePersonality();
    $this->_values['personality_challenge'] = $this->_calculatePersonalityChallenge();
    $this->_values['expression'] = $this->_calculateExpression();
    $this->_values['expression_challenge'] = $this->_calculateExpressionChallenge();
    $this->_values['inclusion_table'] = $this->_calculateInclusionTable();
    $this->_values['first_reponse'] = $this->_calculateFirstResponse();
    $this->_values['seperate_transits'] = $this->_calculateSeperateTransits($this->_translated_birth_name);
    $this->_values['transits'] = $this->_calculateTransitTable();
  }

  private function searchMultipleInArray ($needle, $haystack)
  {
    $first_occurance = count($haystack);
    foreach ($needle as $value)
    {
      $a = array_search($value, $haystack);
      if ($a !== FALSE)
      {
        if($a<$first_occurance)
        {
          $first_occurance = $a;
        }
      }
    }

    if ($first_occurance==count($haystack))
    {
      return FALSE;
    }
    return $first_occurance;
  }

  protected function _calculateNicknameValue ()
  {
    $raw_value = NumberHelpers::reduceStringToValue($this->_translated_current_name);
    return NumberHelpers::reduceNumber($raw_value, array(11,22));
  }

  protected function _calculateNameValues ()
  {
    $values = array();
    // Calculate the total
    $values['total'] = NumberHelpers::reduceNumber(
      NumberHelpers::reduceStringToValue($this->_translated_birth_name),
      array(11,22)
      );

    // Start by splitting the name up into 3 sections
    $name_array = explode(' ', $this->_translated_birth_name);
    $first_name = array_shift($name_array);
    $last_name = array_pop($name_array);
    if (count($name_array)>0)
    {
      $middle_names = implode(' ', $name_array);
      $values['middle_names_value'] = NumberHelpers::reduceNumber(
        NumberHelpers::reduceStringToValue($middle_names),
        array(11,22)
      );
    }
    else
    {
      $values['middle_names_value'] = 0;
    }

    $values['first_name_value'] = NumberHelpers::reduceNumber(
      NumberHelpers::reduceStringToValue($first_name),
      array(11,22)
      );
    // var_dump(NumberHelpers::reduceStringToValue($first_name));exit;

    $values['last_name_value'] = NumberHelpers::reduceNumber(
      NumberHelpers::reduceStringToValue($last_name),
      array(11,22)
      );

    return $values;
  }

  protected function _calculateSoulUrge ()
  {
    $vowel_string = NumberHelpers::getVowelString($this->_translated_birth_name);
    return NumberHelpers::reduceNumber(
        NumberHelpers::reduceStringToValue($vowel_string),
        array(11,22)
      );
  }

  protected function _calculateSoulUrgeChallenge ()
  {
    $first_vowel_key = NumberHelpers::getFirstVowel($this->_translated_birth_name);
    $last_vowel_key = NumberHelpers::getFirstVowel(strrev($this->_translated_birth_name));
    if($first_vowel_key === FALSE && $last_vowel_key === FALSE)
    {
      return 0;
    }
    $a = NumberHelpers::fullReduceNumber($first_vowel_key);
    $b = NumberHelpers::fullReduceNumber($last_vowel_key);
    return abs($a - $b);
  }

  protected function _calculatePersonality ()
  {
    $raw_value = NumberHelpers::reduceStringToValue($this->_translated_birth_name, $this->CONSONANTS);
    return NumberHelpers::fullReduceNumber($raw_value);
  }

  protected function _calculatePersonalityChallenge ()
  {
    $name_array = str_split($this->_translated_birth_name);
    $name_array_r = array_reverse($name_array);
    $first_consonant_key = $this->searchMultipleInArray($this->CONSONANTS, $name_array);
    $last_consonant_key = $this->searchMultipleInArray($this->CONSONANTS, $name_array_r);
    if($first_consonant_key === FALSE && $last_consonant_key === FALSE)
    {
      return 0;
    }
    $a = $this->NUMBER_VALUES[$name_array[$first_consonant_key]];
    $a = NumberHelpers::fullReduceNumber($a);
    $b = $this->NUMBER_VALUES[$name_array_r[$last_consonant_key]];
    $b = NumberHelpers::fullReduceNumber($b);
    return abs($a - $b);
  }

  protected function _calculateExpression ()
  {
    // $raw_value = $this->_calculateSoulUrge() + $this->_calculatePersonality();
    $raw_value = NumberHelpers::reduceStringToValue($this->_birth_name);
    //removed due to newer implementation

    return NumberHelpers::reduceNumber($raw_value, array(11,22));
  }

  protected function _calculateExpressionChallenge ()
  {
    return NumberHelpers::fullReduceNumber(
        $this->_calculatePersonalityChallenge() +
        $this->_calculateSoulUrgeChallenge()
      );
  }
  protected function _calculateInclusionTable ()
  {
    $table = array(
      'count' => array('1'=>0,'2'=>0,'3'=>0,'4'=>0,'5'=>0,'6'=>0,'7'=>0,'8'=>0,'9'=>0)
    );
    $name_array = str_split($this->_translated_birth_name);
    foreach ($name_array as $char)
    {
      if(array_key_exists($char, $this->REDUCED_NUMBER_VALUES))
      {
        $value = NumberHelpers::fullReduceNumber($this->REDUCED_NUMBER_VALUES[$char]);
        $table['count'][$value]++;
      }
    }

    if(count($name_array)<= 12)
    {
      $table['balanced'] = array('1'=>2,'2'=>1,'3'=>1,'4'=>1,'5'=>2,'6'=>1,'7'=>1,'8'=>1,'9'=>2);
    }
    else if (count($name_array)<= 23)
    {
      $table['balanced'] = array('1'=>3,'2'=>2,'3'=>2,'4'=>2,'5'=>2,'6'=>2,'7'=>1,'8'=>2,'9'=>2);
    }
    else
    {
      $table['balanced'] = array('1'=>4,'2'=>3,'3'=>3,'4'=>3,'5'=>3,'6'=>3,'7'=>2,'8'=>3,'9'=>3);
    }
    // var_dump($table);exit;
    return $table;
  }

  protected function _calculateFirstResponse ()
  {
    $table = $this->_calculateInclusionTable();
    $num_zeros = 9;
    foreach ($table['count'] as $value)
    {
      if ($value == 0)
      {
        $num_zeros--;
      }
    }

    return $num_zeros;
  }

  protected function _calculateTransitTable ($previous = 0)
  {
    $name_array = str_split($this->_translated_birth_name);
    $transit_array = array();
    foreach ($name_array as $char)
    {
      if (array_key_exists($char, $this->NUMBER_VALUES))
      {
        $value = NumberHelpers::fullReduceNumber($this->NUMBER_VALUES[$char]);
        $previous += $value;
        $transit_array []= array(
          'letter' => $char,
          'end' => $previous,
          'value' => $value
        );
      }
    }

    return $transit_array;


    /*//new code
    $names = $this->getSeperateTransits();
    $full_name = array_merge($names[0],array_merge($names[1],$names[2]));
    foreach ($full_name as $char){
      $previous += $char['value'];
      $char['end'] = $previous;
    }
    return $full_name;*/
  }

  protected function _calculateSeperateTransits($previous = 0)
  {
    // Start by splitting the name up into 3 sections
    $name_array = explode(' ', $this->_translated_birth_name);
    $first_name = array_shift($name_array);
    $last_name = array_pop($name_array);
    // var_dump($first_name,$last_name,$name_array);exit;
    if (count($name_array)>0)
    {
      $middle_names = implode(' ', $name_array);
      $middle_name_transits = $this->_calculateNameTransits($middle_names, $previous);
    }
    else
    {
      // var_dump("Hello there");exit;
      $middle_name_transits = null;
    }

    $transits = array();
    //$transit_cycle = 0;
    $first_name_transits = $this->_calculateNameTransits($first_name, $previous);
    // var_dump($first_name_transits);exit;
    $last_name_transits = $this->_calculateNameTransits($last_name, $previous);
    // var_dump($last_name_transits);exit;

    $transits = array(
      'first_name' => $first_name_transits,
      'middle_name' => $middle_name_transits,
      'last_name' => $last_name_transits
      );
    return $transits;
  }

  protected function _calculateNameTransits($name, $previous = 0)
  {
    $name_array = str_split($name);
    $transit_array = array();

    foreach($name_array as $char)
    {
      if (array_key_exists($char, $this->NUMBER_VALUES))
      {
        $value = NumberHelpers::fullReduceNumber($this->NUMBER_VALUES[$char]);
        $previous += $value;
        $transit_array []= array(
          'letter' => $char,
          'value' => $value,
          'end' => $previous
          );
        //$transit_cycle += $value;
      }
    }

    return $transit_array;
  }

  public function getNicknameValue ()
  {
    return $this->_values['nickname_value'];
  }

  public function getBirthFirstNameValue ()
  {
    return $this->_values['first_name_value'];
  }

  public function getBirthMiddleNamesValue ()
  {
    return $this->_values['middle_names_value'];
  }

  public function getBirthLastNameValue ()
  {
    return $this->_values['last_name_value'];
  }

  public function getSoulUrge ()
  {
    return $this->_values['soulurge'];
  }

  public function getSoulUrgeChallenge ()
  {
    return $this->_values['soulurge_challenge'];
  }

  public function getPersonality ()
  {
    return $this->_values['personality'];
  }

  public function getPersonalityChallenge ()
  {
    return $this->_values['personality_challenge'];
  }

  public function getExpression ()
  {
    return $this->_values['expression'];
  }

  public function getExpressionChallenge ()
  {
    return $this->_values['expression_challenge'];
  }

  public function getInclusionTable ()
  {
    return $this->_values['inclusion_table'];
  }

  public function getFirstResponse ()
  {
    return $this->_values['first_reponse'];
  }

  public function getTransits ()
  {
    return $this->_values['transits'];
  }

  public function getSeperateTransits ()
  {
    return $this->_values['seperate_transits'];
  }

  public function debug ()
  {
    return $this->_values;
  }

  public function getLiteCalculations()
  {
    $calculations = $this->_values;

    // Remove calculations which are rarely needed
    unset(
      $calculations['inclusion_table'],
      $calculations['first_reponse'],
      $calculations['seperate_transits'],
      $calculations['transits']
    );

    return $calculations;
  }


}
