<?php
/**
 * @author  Daniel J Holmes dan@djcentric.com
 * Copyright (c) 2014, Daniel J Holmes
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by Centric Web Estate.
 * 4. Neither the name of the Centric Web Estate nor the
 *    names of its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 5. This product cannot be used in a commercial environment without the 
 *    express written permission of the copyright holder.
 *
 * THIS SOFTWARE IS PROVIDED BY DANIEL J HOLMES ''AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL DANIEL J HOLMES BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE SITE NUMEROLOGIST.COM AND ITS OWNER HAVE BEEN PROVIDED WITH EXPRESS 
 * WRITTEN PERMISSION FOR THE USE OF THIS PRODUCT AND COPYING ITS CONTENTS IN
 * FULL BUT NOT IN PART. THIS PRODUCT MAY BE COPIED TO MULTIPLE LOCATIONS ON
 * THE SERVER HOSTING NUMEROLOGIST.COM AND MAY NOT BE COPIED ELSEWHERE WITHOUT
 * EXPRESS WRITTEN PERMISSION.
 */

/* vim: set expandtab tabstop=2 shiftwidth=2 softtabstop=2 foldmethod=marker: */


/**
 * List of calulcation algorithms
 * 'MMDDYYYY' => array(
 *   'life-path' 								=> fullReduce((DD + MM) + YYYY) = assert(1-9 || 11 || 22) 
 *   'compound' 								=> reduce(DD+MM+YYYY) = assert(4-36) (NOTE: not a full reduction)
 *   'formative-cycle' 					=> fullReduce(MM) = assert(1-9 || 11)
 *   'productive-cycle' 				=> fullReduce(DD) = assert(1-9 || 11 || 22) 
 *   'productive-cycle-start' 	=>
 *   'harvest-cycle' 						=> fullReduce(YYYY) = assert(1-9 || 11 || 22)
 *   'birthday-number' 					=> DD
 *   'birthday-vibration' 			=> fullReduce(DD) = assert(1-9 || 11 || 22),
 *   'birthday-challenge' 			=> abs(mm-dd) = assert(if(10-31 && not 11 or 22){ where (number is xy){ abs(x-y) } } )  
 *   'birthday-gift' 						=> abs(9 - birthday-challenge) 
 *   'pinnacle-one-end' 				=> 36 - lifepath
 *   'pinnacle-one-value' 			=> fullReduce(reduce(MM) + reduce(DD)) = assert(1-9 || 11 || 22)
 *   'pinnacle-two-end' 				=> pinnacle-one-end + 9
 *   'pinnacle-two-value' 			=> fullReduce(reduce(DD) + reduce(YYYY)) = assert(1-9 || 11 || 22)
 *   'pinnacle-three-end' 			=> pinnacle-two-end + 9
 *   'pinnacle-three-value' 		=> fullReduce(pinnacle-one-value + pinnacle-two-value)  = assert(1-9 || 11 || 22)
 *   'pinnacle-four-value' 			=> (reduce(MM) + reduce(YYYY)) = assert(1-9 || 11 || 22)
 *   'minor-challenge-one' 			=> abs(reduce(MM) - reduce(DD)) = assert(0-9) 
 *   'minor-challenge-two' 			=> abs(reduce(DD) - reduce(YYYY)) = assert(0-9)
 *   'major-challenge' 					=> abs(minor-challenge-one - minor-challenge-two) = assert(0-9)
 *   'achievement' 							=> fullReduce(DD + MM) = assert(1-9 || 11 || 13 || 14 || 16 || 19 || 22 || 33)
 *   'current-personal-year' 		=> fullReduce((DD + MM) + CYYY) = assert(1-9)
 *   'personal-year-yyyy' 			=> The current year the dataset is set to check
 *   'current-personal-month' 	=> fullReduce(current-personal-year + MM)
 *   'personal-month-mm' 				=> The month used in the dataset to check against
 *   'current-personal-day' 		=> fullReduce(current-personal-month + DD)
 *   'personal-day-dd' 					=> The day of the month used to calculate the personal day
 * )
 *
 */

class LifePathCalculator
{
	protected $_birth_date;
	protected $_calculations;

	/**
	 * Initialise the LifePathCalculator
	 *
	 * Expects to be initialised with an SplFixedArray
	 * @param SplFixedArray $birth_date Expects to be in the format MM, DD, YYYY
	 */
	public function __construct (\SplFixedArray $birth_date)
	{
		if ($birth_date->getSize() === 3)
		{
			$this->_birth_date = $birth_date;
		}
		else
		{
			throw new \Exception('Invalid Fixed Array Size');
		}


		$this->_calculate();
	}

	public function getBirthDate ()
	{
		return $this->_birth_date;
	}

	private function _calculate()
	{
		/**
		 * NOTE: The nature and amount of calculations that need to be undertaken mean that
		 * its safer to do each calculation individually. What will be lost in speed will be
		 * gained in accuracy and robustness in that if one function is changed it will be
		 * much easier to debug.
		 */
		
		$this->_calculations = array(
			'life_path' 								=> $this->_calculateLifePath(),
			'compound' 									=> $this->_calculateCompound(),
			'formative_cycle' 					=> $this->_calculateFormativeCycle(),
			'productive_cycle' 					=> $this->_calculateProductiveCycle(),
			'harvest_cycle' 						=> $this->_calculateHarvestCycle(),
			'birthday_number'						=> (int) $this->_birth_date[1],
			'birthday_vibration'				=> $this->_calculateBirthdayVibration(),
			'birthday_challenge'				=> $this->_calculateBirthdayChallenge(),
			'birthday_gift'							=> $this->_calculateBirthdayGift(),
			'pinnacle_one_end'					=> $this->_calculatePinnacleOneEnd(),
			'pinnacle_two_end'					=> $this->_calculatePinnacleTwoEnd(),
			'pinnacle_three_end'				=> $this->_calculatePinnacleThreeEnd(),
			'pinnacle_one_value'				=> $this->_calculatePinnacleOne(),
			'pinnacle_two_value'				=> $this->_calculatePinnacleTwo(),
			'pinnacle_three_value'			=> $this->_calculatePinnacleThree(),
			'pinnacle_four_value'				=> $this->_calculatePinnacleFour(),
			'minor_challenge_one'				=> $this->_calculateMinorChallengeOne(),
			'minor_challenge_two'				=> $this->_calculateMinorChallengeTwo(),
			'major_challenge'						=> $this->_calculateMajorChallenge(),
			'achievement'								=> $this->_calculateAchievement(),
			'achievement_reduced'				=> $this->_calculateReducedAchievement()
			// personal year cannot be stored
			// personal month cannot be stored
			// personal day cannot be stored
		);
	}

	/** PRIVATE CALCULATIONS */

	protected function _calculateLifePath () {
		// Formula
		// reduce(reduce(DD + MM) + YYYY) = assert(1-9 || 11 || 22)
		
		$reduced_MM = NumberHelpers::fullReduceNumber($this->_birth_date[0]);
		$reduced_DD = NumberHelpers::fullReduceNumber($this->_birth_date[1]);
		$to_reduce = $reduced_DD + $reduced_MM + $this->_birth_date[2];
		return NumberHelpers::reduceNumber($to_reduce, array(11,22));
	}

	protected function _calculateCompound () {
		// Formula
		// reduce(DD+MM+YYYY) = assert(4-36)
		$valid_values = array();
		for ($i=10;$i<=28;$i++) {
			switch ($i)
			{
				case 11:
				case 22:
					break;
				default:
					$valid_values []= $i;					
					break;
			}
		}

		$to_reduce = $this->_birth_date[0] + $this->_birth_date[1] + $this->_birth_date[2];
		return NumberHelpers::reduceNumber($to_reduce, $valid_values);
	}

	protected function _calculateFormativeCycle () {
		// Formula
		// fullReduce(MM) = assert(1-9 || 11)
		
		return NumberHelpers::reduceNumber($this->_birth_date[0], array(11));
	}

	protected function _calculateProductiveCycle () {
		// Formula
		// fullReduce(DD) = assert(1-9 || 11 || 22)
		
		return NumberHelpers::reduceNumber($this->_birth_date[1], array(11, 22));
	}

	protected function _calculateHarvestCycle () {
		// Formula
		// fullReduce(YYYY) = assert(1-9 || 11 || 22)
		
		return NumberHelpers::reduceNumber($this->_birth_date[2], array(11, 22));
	}

	protected function _calculateBirthdayVibration () {
		// Formula
		// fullReduce(DD) = assert(1-9)
		// Same formulae as _calculateProductiveCycle
		
		return NumberHelpers::fullReduceNumber($this->_birth_date[1]);
	}

	protected function _calculateBirthdayChallenge () {
		// Formula
		// abs(mm-dd)

		$number = abs($this->_birth_date[0] - $this->_birth_date[1]);
		// Commented out as I misinterpreted the formula
		// if ($number > 9 && $number < 32 && $number !== 11 && $number !== 22)
		// {
			
		// 	if (strlen((string) $number) > 1)
		// 	{
		// 		$split = str_split((string) $number);
		// 		return abs($split[0] - $split[1]);
		// 	}

		// 	return $number;
			
		// }

		return $number;
	}

	protected function _calculateBirthdayGift () {
		// Formula
		// abs(9 - birthday-challenge)
		
		$birthday_challenge = $this->_calculateBirthdayChallenge();
		if($birthday_challenge !== false) {
			return abs(9 - $birthday_challenge);
		}

		return false;
	}

	protected function _calculatePinnacleOneEnd () {
		// Formula
		// 36 - lifepath
		
		$lifepath = $this->_calculateLifePath();
		return 36 - $lifepath;
	}

	protected function _calculatePinnacleTwoEnd () {
		// Formula
		// pinnacle-one-end + 9
		$var = $this->_calculatePinnacleOneEnd() + 9;
		return $var;
	}

	protected function _calculatePinnacleThreeEnd () {
		// Formula
		// pinnacle-two-end + 9
		
		$var = $this->_calculatePinnacleTwoEnd() + 9;
		return $var;
	}

	protected function _calculatePinnacleOne () {
		// Formula
		// fullReduce(reduce(MM) + reduce(DD)) = assert(1-9)
		
		$reduced_MM = NumberHelpers::fullReduceNumber($this->_birth_date[0]);
		$reduced_DD = NumberHelpers::fullReduceNumber($this->_birth_date[1]);
		return NumberHelpers::fullReduceNumber($reduced_MM + $reduced_DD);
	}

	protected function _calculatePinnacleTwo () {
		// Formula
		// fullReduce(reduce(DD) + reduce(YYYY)) = assert(1-9)
		
		$reduced_YYYY = NumberHelpers::fullReduceNumber($this->_birth_date[2]);
		$reduced_DD = NumberHelpers::fullReduceNumber($this->_birth_date[1]);
		return NumberHelpers::fullReduceNumber($reduced_DD + $reduced_YYYY);
	}

	protected function _calculatePinnacleThree () {
		// Formula
		// fullReduce(pinnacle-one-value + pinnacle-two-value)  = assert(1-9)
		
		$input = $this->_calculatePinnacleOne() + $this->_calculatePinnacleTwo();
		return NumberHelpers::fullReduceNumber($input);
	}

	protected function _calculatePinnacleFour () {
		// Formula
		// reduce(reduce(MM) + reduce(YYYY)) = assert(1-9)
		
		$reduced_YYYY = NumberHelpers::fullReduceNumber($this->_birth_date[2]);
		$reduced_MM = NumberHelpers::fullReduceNumber($this->_birth_date[0]);
		return NumberHelpers::fullReduceNumber($reduced_MM + $reduced_YYYY);
	}

	protected function _calculateMinorChallengeOne () {
		// Formula
		// abs(reduce(MM) - reduce(DD)) = assert(0-8) 
		
		$reduced_MM = NumberHelpers::fullReduceNumber($this->_birth_date[0]);
		$reduced_DD = NumberHelpers::fullReduceNumber($this->_birth_date[1]);
		return abs($reduced_DD - $reduced_MM);
	}

	protected function _calculateMinorChallengeTwo () {
		// Formula
		// abs(reduce(DD) - reduce(YYYY)) = assert(0-8)
		
		$reduced_YYYY = NumberHelpers::fullReduceNumber($this->_birth_date[2]);
		$reduced_DD = NumberHelpers::fullReduceNumber($this->_birth_date[1]);
		return abs($reduced_DD - $reduced_YYYY);
	}

	protected function _calculateMajorChallenge () {
		// Formula
		// abs(minor-challenge-one - minor-challenge-two) = assert(0-8)
		
		$minor_challenge_one = $this->_calculateMinorChallengeOne();
		$minor_challenge_two = $this->_calculateMinorChallengeTwo();
		return abs($minor_challenge_one - $minor_challenge_two);
	}

	protected function _calculateAchievement () {
		// Formula
		// reduce(DD + MM) = assert(1-9 || 11 || 13 || 14 || 16 || 19 || 22 || 33)
		
		return (int) NumberHelpers::reduceNumber(
				$this->_birth_date[1] + $this->_birth_date[0],
				array(11, 13, 14, 16, 19, 22, 33)
			);
	}

	protected function _calculateReducedAchievement () {
		// Formula
		// fullReduce(DD + MM) = assert(1-9)
		
		return NumberHelpers::fullReduceNumber(
				$this->_birth_date[1] + $this->_birth_date[0]
			);
	}

	protected function _calculatePersonalYear ($year) {
		// Formula
		// fullReduce(achievement + CYYY) = assert(1-9)

		$achievement = $this->_calculateAchievement();
		return NumberHelpers::fullReduceNumber($achievement + $year);
	}

	protected function _calculatePersonalMonth ($year, $month) {
		// Formula
		// fullReduce(current-personal-year + MM)
		
		$personal_year = $this->_calculatePersonalYear($year);
		return NumberHelpers::fullReduceNumber($personal_year + $month);
	}

	protected function _calculatePersonalDay ($year, $month, $day) {
		// Formula
		// fullReduce(current-personal-year + MM)
		
		$personal_month = $this->_calculatePersonalMonth($year, $month);
		return NumberHelpers::fullReduceNumber($personal_month + $day);
	}
 
	/** PUBLIC ACCESSORS */
	public function getLifePath () {
		return $this->_calculations['life_path'];
	}

	public function getCompound() {
		return $this->_calculations['compound'];
	}

	public function getFormativeCycle () {
		return $this->_calculations['formative_cycle'];
	}

	public function getProductiveCycle () {
		return $this->_calculations['productive_cycle'];
	}

	public function getHarvestCycle () {
		return $this->_calculations['harvest_cycle'];
	}

	public function getBirthdayNumber () {
		return $this->_calculations['birthday_number'];
	}

	public function getBirthdayVibration () {
		return $this->_calculations['birthday_vibration'];
	}

	public function getBirthdayChallenge () {
		return $this->_calculations['birthday_challenge'];
	}

	public function getBirthdayGift () {
		return $this->_calculations['birthday_gift'];
	}

	public function getFirstPinnacleEnding () {
		return $this->_calculations['pinnacle_one_end'];
	}

	public function getSecondPinnacleEnding () {
		return $this->_calculations['pinnacle_two_end'];
	}

	public function getThirdPinnacleEnding () {
		return $this->_calculations['pinnacle_three_end'];
	}

	public function getFirstPinnacle () {
		return $this->_calculations['pinnacle_one_value'];
	}

	public function getSecondPinnacle () {
		return $this->_calculations['pinnacle_two_value'];
	}

	public function getThirdPinnacle () {
		return $this->_calculations['pinnacle_three_value'];
	}

	public function getFourthPinnacle () {
		return $this->_calculations['pinnacle_four_value'];
	}

	public function getFirstMinorChallenge () {
		return $this->_calculations['minor_challenge_one'];
	}

	public function getSecondMinorChallenge () {
		return $this->_calculations['minor_challenge_two'];
	}

	public function getMajorChallenge () {
		return $this->_calculations['major_challenge'];
	}

	public function getAchievement () {
		return $this->_calculations['achievement'];
	}

	public function getReducedAchievement () {
		return $this->_calculations['achievement_reduced'];
	}

	public function getPersonalYearFor ($year) {
		return $this->_calculatePersonalYear($year);
	}

	public function getPersonalMonthFor ($month, $year) {
		return $this->_calculatePersonalMonth($year, $month);
	}

	public function getPersonalDayFor ($month, $day, $year) {
		return $this->_calculatePersonalMonth($year, $month, $day);
	}

	public function debug ()
	{
		return $this->_calculations;
	}
}
